import java.util.*;
import java.io.*;

/**
 * This class contains the methods for constructing a phylogenetic tree from a 
 * Newick tree representation
 *
 * @author Vivek Jayaswal
 *
 */
class NewickTreeTraversal {	

	boolean showOutput;
	int internalNodeCount = 1;
	LinkedList intNodeToLeafNode;
	TreeMap phylogeneticTree;
	
	//Default Constructor
	NewickTreeTraversal() {
		showOutput = true;	
	}	
	
	//Constructor called from other programs
	NewickTreeTraversal(boolean displayOutput) {
		showOutput = displayOutput;	
	}
	
	public static void main(String[] args) {

		TreeMap newickTree = new TreeMap();		
		String treeNotation = args[0];
		ArrayList leafOrInternalNodesList;
		
		NewickTreeTraversal ntt = new NewickTreeTraversal(true);
		
		// create tree
		ntt.generateTree(treeNotation, newickTree);	
		
		//display tree
		ntt.displayTree();
		
		//identify all leaf nodes
		leafOrInternalNodesList = ntt.determineLeafNodes(newickTree);
		
		//compute path to root from each leaf node 
		for(int i=0; i<leafOrInternalNodesList.size(); i++) {
			String node = (String)leafOrInternalNodesList.get(i);			
			ntt.leafToRootPath(node, newickTree);
		}
		
		leafOrInternalNodesList = ntt.determineInternalNodes(newickTree);
		
		//determine root node
		String rootNode = ntt.getRootNode(newickTree);

		NodeStructure ns = (NodeStructure) newickTree.get(rootNode);
		String newickString = ntt.createNewickTree(newickTree, ns.getChildNode1());
		
		//Add the root node to the generated string
		newickString = "(" + rootNode.substring(0,rootNode.lastIndexOf('-')) + ","
						+ newickString + ")";
		
		//Obtain the Date and time (HH:MM:SS)
		Calendar calendar = Calendar.getInstance();
		newickString += "_" + calendar.get(Calendar.DATE) 
						+ "_" + calendar.get(Calendar.HOUR_OF_DAY)
						+ ":" + calendar.get(Calendar.MINUTE)
						+ ":" + calendar.get(Calendar.SECOND);
						
		System.out.println("Re-generated Newick Format Tree = " + newickString);

		//ShowPhylogeneticTree sPTree 
		//	= new ShowPhylogeneticTree(newickTree, rootNode, "C:/javapgms/Tree.jpg");

	}
	
	/**	This method is used to generate a phylogenetic tree network
	 *
	 *	@param notation			Newick Tree
	 */
	void generateTree(String notation, TreeMap pTree) {

		//Initialization
		phylogeneticTree = pTree;
		intNodeToLeafNode = new LinkedList();
		
		// A stack of elements present at each level	 
		TreeMap levelMap = new TreeMap();		
		
		LinkedList levelOneNodes = new LinkedList();
		LinkedList neighbourNodes;
		
		String nodeOnStack = ""; 
		String internalNodeName = "";
		String leafNodeName = "";

		int treeLength = notation.length();
		int treeLevelCount = 0;
						
		//Choose current and previous symbols such that they are not equal
		char currSymbol = '#';	
		char prevSymbol = '#';
		
		for(int i=0; i<treeLength; i++) {

			prevSymbol = currSymbol;			
			currSymbol = notation.charAt(i);
			
			// If current symbol = "(", increment the tree level count
			if(currSymbol == '(') {
				
				/* Check whether the prevSymbol was an alphabet. If yes, create
				 * a leaf node for the last read sequence name
				 */
				if(Character.isLetterOrDigit(prevSymbol)) {	

					leafNodeName = leafNodeName + "-Node";
					createNode(leafNodeName, "LEAF");
					
					neighbourNodes = new LinkedList();
					neighbourNodes.add(leafNodeName);
					
					//add the leaf node to the new level
					levelMap.put(new Integer(treeLevelCount),neighbourNodes);
					
					if(treeLevelCount == 1) {
						levelOneNodes.add(leafNodeName);
					}
				
					//re-initialize
					leafNodeName = "";					
						
				} //leaf node found				
				
				treeLevelCount++;
				 
			}// end of IF stmt for current symbol = '('
			
			// If current symbol = ")", decrement the tree level count
			if(currSymbol == ')') {

				// If tree level count > 1, connect the internal nodes/ leaf nodes
				if(treeLevelCount > 1) {
					
					Integer levelNumber = new Integer(treeLevelCount);
					
					/* Check whether the prevSymbol was an alphabet. If yes, create
					 * a leaf node for the last read sequence name
					 */
					if(Character.isLetterOrDigit(prevSymbol)) {	
					
						leafNodeName = leafNodeName + "-Node";
						createNode(leafNodeName, "LEAF");
						
						//get the neighbouring node
						neighbourNodes = (LinkedList)levelMap.get(levelNumber);
						
						//Link the current leaf node and the neighbour node
						nodeOnStack = (String)neighbourNodes.get(0);							 
						internalNodeName = linkToInternalNode(nodeOnStack, leafNodeName);
						
						//delete the entry for the current tree level
						levelMap.remove(levelNumber);
						
						// add the internal node to the list of nodes at the next
						// level
						levelNumber = new Integer(treeLevelCount-1); 
						
						if(levelMap.containsKey(levelNumber)) {
							neighbourNodes = (LinkedList)levelMap.get(levelNumber);
							neighbourNodes.add(internalNodeName);	
						}	
						else {
							neighbourNodes = new LinkedList();
							neighbourNodes.add(internalNodeName);
							levelMap.put(levelNumber, neighbourNodes);
						}						
												
						//re-initialze leaf node value
						leafNodeName = "";
												
					} //leaf node found				
					else {
						
						//obtain the list of neighbouring nodes for the given level
						neighbourNodes = (LinkedList)levelMap.get(levelNumber);
						
						//add the neighbouring nodes together
						internalNodeName = linkToInternalNode(
													(String)neighbourNodes.get(0)
													,(String)neighbourNodes.get(1));
						
						//delete the entry for the current tree level
						levelMap.remove(levelNumber);
						
						// add the internal node to the list of nodes at the next
						// level
						levelNumber = new Integer(treeLevelCount-1); 
						
						if(levelMap.containsKey(levelNumber)) {
							neighbourNodes = (LinkedList)levelMap.get(levelNumber);
							neighbourNodes.add(internalNodeName);	
						}	
						else {
							neighbourNodes = new LinkedList();
							neighbourNodes.add(internalNodeName);
							levelMap.put(levelNumber, neighbourNodes);
						}					
						
					}				
				
					treeLevelCount--;	
					
					//Add to the list of top level nodes
					if(treeLevelCount == 1) {
						levelOneNodes.add(internalNodeName);
					}
				} // tree level count > 1
				else {

					/* Check whether the prevSymbol was an alphabet. If yes, create
					 * a leaf node for the last read sequence name
					 */
					if(Character.isLetterOrDigit(prevSymbol)) {		
										
						leafNodeName = leafNodeName + "-Node";
						createNode(leafNodeName, "LEAF");
						
						levelOneNodes.add(leafNodeName);
						
						//re-initialize
						leafNodeName = "";
					}
					
					//join the level 1 nodes
					//System.out.println(phylogeneticTree);
					connectLevelOneNodes(levelOneNodes);
					
				} // Tree count = 1			
				
			} // end of IF stmt for current symbol = ')'
			
			// If current symbol is an alphabet
			if(Character.isLetterOrDigit(currSymbol)) {												
				leafNodeName += currSymbol;
			}
			
			// If current symbol is ','
			if(currSymbol == ',') {
				
				// Determine whether a leaf node has been read
				if(Character.isLetterOrDigit(prevSymbol)) {		
				
					leafNodeName = leafNodeName + "-Node";
					createNode(leafNodeName, "LEAF");					
					
					neighbourNodes = new LinkedList();
					neighbourNodes.add(leafNodeName);
					
					//Add the tree to the current level					
					levelMap.put(new Integer(treeLevelCount), neighbourNodes);
					
					//Add to the list of top level nodes
					if(treeLevelCount == 1) {
						levelOneNodes.add(leafNodeName);
					}
					
					//re-initialize
					leafNodeName = "";
				}
												
			} // end of IF stmt for current symbol = ','
			
		} // end of FOR loop
			
	}
	
	/** 
	 * This method creates a new node and attaches it to the phylogenetic tree
	 *
	 * @param node				Name of the node
	 * @param nodeType			Type of node ("LEAF", "INTERNAL")
	 */ 
	void createNode(String node, String nodeType) {
		
		NodeStructure ns = new NodeStructure();		
		ns.setNodeType(nodeType);		
		phylogeneticTree.put(node, ns);	
		
	}
	
	/**	This method is used to link two nodes together via an internal node
	 *
	 *	@param	node1			Node 1 (Leaf/ Internal node)
	 *	@param	node2			Node 2 (Leaf/ Internal node)
	 *
	 *	@return	String			Internal node name
	 */
	String linkToInternalNode(String node1, String node2) {
		
		String intNode = internalNodeCount + "-Node";

		//create a new internal node
		createNode(intNode, "INTERNAL");		
		internalNodeCount++;
						
		//Link node1 and node2 to the internalNode
		linkNodes(node1, intNode);
		linkNodes(node2, intNode);
				
		return intNode;		
	}
	
	/**
	 * This method links a child node to its parent node
	 * 
	 * @param childNodeName		Name of child node
	 * @param parentNodeName 	Name of parent node
	 */ 
	void linkNodes(String childNodeName, String parentNodeName) {
		int i;
		
		NodeStructure child = 
			(NodeStructure)phylogeneticTree.get(childNodeName);
			
		NodeStructure parent = 
			(NodeStructure)phylogeneticTree.get(parentNodeName);
		
		//attach child node to parent node
		child.setParentNode(parentNodeName);		
		phylogeneticTree.put(childNodeName, child);
		
		/*determine whether child node is attached to childNode1 or childNode2
		 *variable of parent
		 */		
		i = parent.getChildNodeCount();				

		//attach parent node to child node
		switch(i) {
			case 0: parent.setChildNode1(childNodeName);
					break;
			case 1: parent.setChildNode2(childNodeName);
		}								

		//Increment the parent node's child count
		parent.setChildNodeCount(i + 1);				
		
		phylogeneticTree.put(parentNodeName, parent);				
	}
	
	/**	This method is used to connect the nodes at topmost level. The number
	 *	of nodes could be two or three
	 *
	 *	@param	nodesList			Linked List of nodes at the topmost level
	 */
	void connectLevelOneNodes(LinkedList nodesList) {
		int listSize = nodesList.size();
		String[] nodeNames = new String[3];
		String rootNode = null;
		String childNode = "";
		
		NodeStructure ns;
		
		if(listSize == 2) {
			nodeNames[0] = (String)nodesList.get(0);
			nodeNames[1] = (String)nodesList.get(1);
			
			//Determine whether one of the nodes is a leaf node
			ns = (NodeStructure)phylogeneticTree.get(nodeNames[0]);
			if((ns.getNodeType()).equals("LEAF")) {
				rootNode = nodeNames[0];
				childNode = nodeNames[1];
			}
			
			ns = (NodeStructure)phylogeneticTree.get(nodeNames[1]);
			if((ns.getNodeType()).equals("LEAF")) {
				rootNode = nodeNames[1];
				childNode = nodeNames[0];
			}	
			
			if(rootNode == null) {
				ns = (NodeStructure)phylogeneticTree.get(nodeNames[0]);				
				ns.setParentNode(nodeNames[1]);
				phylogeneticTree.put(nodeNames[0], ns);
				
				//find root node within the sub-tree having nodeNames[1] as the root
				realignNodeSet(nodeNames[1], nodeNames[0]);
				
			}
			else {
				//link the internal node set to the leaf node
				ns = (NodeStructure)phylogeneticTree.get(rootNode);
				ns.setChildNode1(childNode);
				ns.setChildNode2(null);
				ns.setNodeType("ROOT");
				phylogeneticTree.put(rootNode, ns);
				
				ns = (NodeStructure)phylogeneticTree.get(childNode);
				ns.setParentNode(rootNode);
				phylogeneticTree.put(childNode, ns);				
			}
		} // list size = 2
		
		if(listSize == 3) {
			nodeNames[0] = (String)nodesList.get(0);
			nodeNames[1] = (String)nodesList.get(1);
			nodeNames[2] = (String)nodesList.get(2);
			
			String internalNode = internalNodeCount + "-Node";
			internalNodeCount++;
			
			createNode(internalNode, "INTERNAL");
			
			//identify root node
			int rootNodeNumber = -1;			
			
			ns = (NodeStructure)phylogeneticTree.get(nodeNames[0]);
			if((ns.getNodeType()).equals("LEAF")) {
				rootNode = nodeNames[0];
				rootNodeNumber = 0;
			}

			ns = (NodeStructure)phylogeneticTree.get(nodeNames[1]);
			if((ns.getNodeType()).equals("LEAF")) {
				rootNode = nodeNames[1];
				rootNodeNumber = 1;
				
				//swap nodes 1 and 0
				String temp = nodeNames[0];
				nodeNames[0] = rootNode;
				nodeNames[1] = temp;
			}
			
			ns = (NodeStructure)phylogeneticTree.get(nodeNames[2]);
			if((ns.getNodeType()).equals("LEAF")) {
				rootNode = nodeNames[2];
				rootNodeNumber = 2;
		
				//swap nodes 0 and 2
				String temp = nodeNames[0];
				nodeNames[0] = rootNode;
				nodeNames[2] = temp;

			}
															
			if(rootNodeNumber != -1) {
				linkNodes(internalNode, rootNode);
				
				ns = (NodeStructure)phylogeneticTree.get(rootNode);
				ns.setChildNode1(internalNode);
				ns.setChildNode2(null);
				ns.setNodeType("ROOT");
				phylogeneticTree.put(rootNode, ns);
				
				//link the two internal nodes
				linkNodes(nodeNames[1], internalNode);
				linkNodes(nodeNames[2], internalNode);
				
			} // root exists
			else {
				//linkNodes(String childNodeName, String parentNodeName)
				linkNodes(nodeNames[0], internalNode);
				linkNodes(nodeNames[1], internalNode);
			
				//find root node within the sub-tree having nodeNames[1] as the root	
				realignNodeSet(nodeNames[2], internalNode);		
			} // root does not exist
			
		} //list size = 3
	}
	
	/**	This method is used to determine the path from a previously selected
	 *	internal node to one of the leaf nodes linked to the internal node
	 *
	 *	@param	node1				Current node (leaf node/ internal node)
	 *	@param	origChildNode		Internal node that is to be converted
	 *								from parent node to child node
	 */
	void realignNodeSet(String node1, String origChildNode) {
		
		NodeStructure ns = (NodeStructure)phylogeneticTree.get(node1);
		intNodeToLeafNode.add(node1);
		
		if((ns.getNodeType()).equals("LEAF")) {
			ns.setNodeType("ROOT");	
			phylogeneticTree.put(node1, ns);
					
			modifyNodeLinks(origChildNode);
		}
		else {
			String cNode = ns.getChildNode1();
			if(cNode != null) realignNodeSet(cNode, origChildNode);
			else realignNodeSet(ns.getChildNode2(), origChildNode);
		}
		
	}

	/**	This method is used to reverse the direction of traversal between a 
	 *	previously defined internal node and leaf node
	 *
	 *	@param	cNode				Name of the internal node
	 */			
	void modifyNodeLinks(String cNode) {
		String curNode = "";
		String prevNode = "";
		NodeStructure ns;
		
		for(int i=intNodeToLeafNode.size()-1; i>=0; i--) {
			
			prevNode = curNode;
			curNode = (String)intNodeToLeafNode.get(i);
			
			ns = (NodeStructure)phylogeneticTree.get(curNode);
			
			if(i == intNodeToLeafNode.size()-1) {
				ns.setNodeType("ROOT");
				ns.setParentNode(null);
				ns.setChildNode2(null);
				ns.setChildNode1((String)intNodeToLeafNode.get(i-1));
				
			}	
			else {
				ns.setParentNode(prevNode);
				
				if(i > 0) {
					int count = ns.getChildNodeNumber(prevNode);
					
					if(count == 0) ns.setChildNode1((String)intNodeToLeafNode.get(i-1));
					else ns.setChildNode2((String)intNodeToLeafNode.get(i-1));
				}
				else {
					int count = ns.getChildNodeNumber(prevNode);
					
					if(count == 0) ns.setChildNode1(cNode);
					else ns.setChildNode2(cNode);					
				}				
				
			} // Node is not the leaf node
			
			//update the phylogenetic tree
			phylogeneticTree.put(curNode, ns);
			
		} // end of FOR loop
		
		// Create link between cNode and its parent node
		ns = (NodeStructure)phylogeneticTree.get(cNode);
		ns.setParentNode(curNode);
		phylogeneticTree.put(cNode, ns);
		
	}

	// This method is used to display a phylogenetic tree
	void displayTree() {
		
		Set set = phylogeneticTree.entrySet();
		Iterator itr  =set.iterator();
		
		while(itr.hasNext()) {
			Map.Entry me = (Map.Entry)itr.next();
			
			NodeStructure node = (NodeStructure) me.getValue();
			String name = (String)me.getKey();
				
			System.out.println(" ");
			System.out.println("Node Name: " + 	name);
			System.out.print("Parent Node = " + node.getParentNode());
			System.out.print(", Child Node 1 = " + node.getChildNode1());
			System.out.print(", Child Node 2 = " + node.getChildNode2());
			System.out.print(", Node type = " + node.getNodeType());
		}				
	}

	/**
	 * This method is used to determine the path from root node to a given
	 * leaf node
	 *
	 * @param leaf					Leaf node
	 * @param phylogeneticTree		Phylogenetic tree
	 *
	 *	@return String
	 */
	String leafToRootPath(String leaf, TreeMap phylogeneticTree) {
		NodeStructure ns;
		ArrayList intNodes = new ArrayList();
		String leafNode = leaf;
		String path = leafNode;
		String parentNode;
		
		ns = (NodeStructure)phylogeneticTree.get(leafNode);
		parentNode = ns.getParentNode();
		
		while(parentNode != null) {
			path = path + ":" + parentNode;
			
			//if parent node does not exist in intNodes list, add it
			if(intNodes.indexOf(parentNode) == -1)
				intNodes.add(parentNode);
			
			ns = (NodeStructure)phylogeneticTree.get(parentNode);
			parentNode = ns.getParentNode();			
		}		
		
		if(showOutput) {
			System.out.println(" ");
			System.out.println("Path from " + leaf + " to root is ");
			System.out.println(path);
		}
		
		return path;
	}
	
	/**
	 * This method is used to determine all the internal nodes of the tree
	 *
	 * @param phylogeneticTree		Phylogenetic tree
	 */
	ArrayList determineInternalNodes(TreeMap phylogeneticTree) {
		ArrayList ls = new ArrayList();
		
		Set set = phylogeneticTree.entrySet();
		Iterator i = set.iterator();
		Map.Entry me;
		String node;
		int j;
		
		while(i.hasNext()) {
			me = (Map.Entry)i.next();
			node = (String)me.getKey();
			
			node = node.substring(0, node.lastIndexOf('-'));
			
			try {
				j = Integer.parseInt(node);
				ls.add((String)me.getKey());
			} catch(Exception e) {
				node = null; //do nothing
			}
		}
		
		if(showOutput) {
			System.out.println("");
			System.out.print("Internal Nodes: ");
			System.out.println(ls);	
		}
				
		return ls;
	}
	
	/**
	 * This method is used to determine all the leaf nodes of the tree
	 *
	 * @param phylogeneticTree		Phylogenetic tree
	 */
	 ArrayList determineLeafNodes(TreeMap phylogeneticTree) {

		ArrayList ls = new ArrayList();
		
		Set set = phylogeneticTree.entrySet();
		Iterator i = set.iterator();
		Map.Entry me;
		String node;
		int j;
		
		NodeStructure ns;
		
		while(i.hasNext()) {
			me = (Map.Entry)i.next();
			node = (String)me.getKey();
			
			//System.out.println("Node = " + node);
			
			ns = (NodeStructure)phylogeneticTree.get(node);
			if((ns.getNodeType()).equals("LEAF")) {
				ls.add((String)me.getKey());
			}
		}
			
		if(showOutput) {
			System.out.println("");
			System.out.print("Leaf Nodes: ");
			System.out.println(ls);	
		}
				
		return ls;
	 		
	 }	 	

	/**
	 * This method is used to determine the root node
	 *
	 * @param phylogeneticTree		Phylogenetic tree
	 */
	 String getRootNode(TreeMap phylogeneticTree) {

		Set set = phylogeneticTree.entrySet();
		Iterator i = set.iterator();
		Map.Entry me;
		String node;
		int j;
		
		NodeStructure ns;
		
		while(i.hasNext()) {
			me = (Map.Entry)i.next();
			node = (String)me.getKey();
			
			ns = (NodeStructure)phylogeneticTree.get(node);
			if((ns.getNodeType()).equals("ROOT")) {
				return node;
			}
		}
			
		return "ERROR";	 		
	 }	 	
	
	/**	This method returns a Newick format tree
	 *
	 *	@param	pTree		Phylogenetic tree stored in TreeMap data structure
	 *	@param	root		Node whose child nodes will be evaluated
	 *	
	 *	@return String
	 */	
	String createNewickTree(TreeMap pTree, String root) {

		NodeStructure ns = (NodeStructure)pTree.get(root);
		
		String subTree = "";			//sub-tree
				
		String cNode1 = ns.getChildNode1();		
		String cNode2 = ns.getChildNode2();
		
		String cNode1Type = "";
		String cNode2Type = "";

		if(cNode1 != null) {
			ns = (NodeStructure)pTree.get(cNode1);
			cNode1Type = ns.getNodeType();
		}	

		if(cNode2 != null) {
			ns = (NodeStructure)pTree.get(cNode2);
			cNode2Type = ns.getNodeType();			
		}
		
		//Both the child nodes are leaf nodes
		if((cNode1Type.equals(cNode2Type)) && cNode1Type.equals("LEAF")) {
			return ("(" + cNode1.substring(0,cNode1.lastIndexOf('-')) 
						+ "," + cNode2.substring(0,cNode2.lastIndexOf('-'))  + ")");			
		}
		
		//Both the child nodes are Internal nodes
		if((cNode1Type.equals(cNode2Type)) && cNode1Type.equals("INTERNAL")) {
			//subTree = "(" + createNewickTree(pTree, cNode1) 
			subTree = "(" + createNewickTree(pTree, cNode1) + ","
							+ createNewickTree(pTree, cNode2) + ")";
			
			return subTree;
		}
		
		//One child node is a leaf node and the other an internal node
		if(!cNode1Type.equals(cNode2Type)) {
			
			if(cNode1Type.equals("LEAF")) {
				//subTree = "(" + cNode1.substring(0,cNode1.lastIndexOf('-')) 
				subTree = "(" + cNode1.substring(0,cNode1.lastIndexOf('-')) + ","
								+ createNewickTree(pTree, cNode2) + ")";
			}			
			else  {
				//subTree = "(" + cNode2.substring(0,cNode2.lastIndexOf('-')) 
				subTree = "(" + cNode2.substring(0,cNode2.lastIndexOf('-')) + ","
								+ createNewickTree(pTree, cNode1) + ")";
			}			
		}		
		
		return subTree;
	}
		
}
