import java.util.*;
import java.io.*;

/**
 *	@author Vivek Jayaswal
 */

class MaximumAverageLikelihood {
	
	boolean showOutput = true;
	
	int noOfSites; 			// No of DNA sites per sequence
	int noOfIterations;		//current iteration count
	int noOfVariableSites;	//no of unique nucleotide patterns
	int noOfDataSets;
    	int userSpecifiedIterationCount;    //no of iterations to be performed

	int[] unvariedSitesIndex = new int[4];
	int[] unvariedSitesCount = new int[4];

	double[] unvariedSitesProb = new double[4];

	//Invariant sites parameters
	double beta;
	double[] probGivenInv = new double[4];

	//Array for returning the probabilities corresponding to bases
	double[] baseProb = new double[4];	

	//4*4 matrix for storing updated Q values
	double[][] updateMatrix = new double[4][4];
	
      double marginalProbVector[] = new double[4];
	double modifiedMargProb[] = new double[4];
        
	//Q-matrix is not updated in the next iteration if the sum of squares is
	//less than user-defined value
	double minValue;
	
	String srcFilePath; 	//Input files directory
	String destFilePath; 	// O/P file
	String qValFilePath;	//Q-values file
	
	FileWriter fw;			// Pointer for storing Q and L values in a file
	FileWriter fSiteW;		// Pointer for site specific L values

	/* TreeMap branchList stores Q-values along each branch. "Key" value is 
 	 * used to identify the child node. Parent node can be determined from 
 	 * the phylogenetic tree
	 */  
	TreeMap branchList =  new TreeMap();	
	
	//Temporarily store updated Q-values
	TreeMap updateBranchList = new TreeMap();	

	//Store current estimates of Lagrange multipliers		
	TreeMap updateLagMultPerBranch = new TreeMap();

	// Store DNA sequences	
	TreeMap treeSequences = new TreeMap(); 

	/* Store RiXi values for internal nodes for each site. Array size corresponds
	 * to the number of variable sites
	 */
	TreeMap probRiXi[];	 	
	
	/* Store SiXi values for internal nodes for each site. Array size corresponds
	 * to the number of variable sites
	 */
	TreeMap probSiXi[];	
	
	//Create a phylogenetic tree
	TreeMap phylogeneticTree = new TreeMap();	// stores the phylogenetic tree

	//Determine whether to update a particular branch
	HashMap isUpdateReqd = new HashMap();

	//Helper class: TreeTraversal(boolean displayOutput) 
	NewickTreeTraversal t = new NewickTreeTraversal(false);
		
	//Array storing the number of times the i th pattern occurs
	int patternCountArray[];
			
	ArrayList leafNodesList;			//list of all leaf nodes
	ArrayList intNodesList;			//list of all internal nodes
		
	double[][][] siteSpecificParentChildNodeVal;  //array to store site prob

	//Data Structures required for generating random Q-matrices
	NodeStructure nsGen;
	QMatrixGenerator qMatGen = new QMatrixGenerator();

	double[][] qGenArray = new double[4][4];

	MaximumAverageLikelihood() {
		//do nothing
	}
	
	/** This constructor is called from the Genetic Algorithm based tree space
	 *	search program
	 *
	 *	@param	treeNotation		Tree notation in Newick format
	 *	@param	sequenceFilePath	Source directory and name of the input file
	 *	@param	siteNumber			No of sites per sequence
	 */
	MaximumAverageLikelihood(String treeNotation, String sequenceFilePath
							, int siteNumber) {
		
		showOutput = false;
		
		srcFilePath = sequenceFilePath;
		noOfSites = siteNumber;
		
		t.generateTree(treeNotation, phylogeneticTree);
		//String rootNode = t.getRootNode(treeNotation) + "-Node";
		String rootNode = t.getRootNode(phylogeneticTree);
		
		/* Determine all leaf nodes. Read sequences from various files based
		 * on the order in which leaf nodes appear in the Newick tree 
		 * representation from left to right
		 */
		//leafNodesList = t.determineLeafNodes();	
		leafNodesList = t.determineLeafNodes(phylogeneticTree);	
		
		//Read the sequences only once
		readSequences("A", leafNodesList, rootNode);
	}
			
	public static void main (String[] args) {			
		MaximumAverageLikelihood lo = new MaximumAverageLikelihood();
				
		lo.readInputParameters();
	}	
		
	/**	This method is used to initialize global variables
	 */
	void initializeGlobalVariables() {

		noOfIterations = 0;		
	
		updateMatrix = new double[4][4];		
		baseProb = new double[4];		
				
		branchList =  new TreeMap();	
		updateBranchList = new TreeMap();	
		phylogeneticTree = new TreeMap();	// stores the phylogenetic tree
	
		isUpdateReqd = new HashMap();
		
		QMatrixGenerator qMatGen = new QMatrixGenerator();
		
		qGenArray = new double[4][4];

	}
		
	/**
	 * This method performs the following functions
	 *	- read input parameters
	 *	- generate phylogenetic tree based on Newick tree representation
	 *	- determine internal nodes
	 *	- determine leaf nodes
	 *	- initialize joint probability distribution for all branches
	 *	- call methods for computing log likelihood and updating probability
	 *	  distributions 
	 */
	void readInputParameters() {
		
		String treeNotation = null;						
		boolean estimateInvariantSites = false;

		BufferedReader br = new BufferedReader
							(new InputStreamReader(System.in));
        
        String userOption = "1";
		
		String provideQValFile = "1"; //User-specified option for Q-values
		
		System.out.println();
		System.out.println(" ------- INPUT PARAMETERS -------");
		try{
			System.out.print("No of sites: ");
			noOfSites = Integer.parseInt(br.readLine()); 

			System.out.print("No of data sets: ");
			noOfDataSets = Integer.parseInt(br.readLine()); 

			System.out.print("Input file: ");
			srcFilePath = br.readLine();		//Input files directory
			
			provideQValFile = "1";
			qValFilePath = "";
						
			if(provideQValFile.charAt(0) == '2') {
				System.out.print("Q-values initialization file: ");
				qValFilePath = br.readLine();		//Q-values file directory
			}
						
			System.out.print("Output file: ");
			destFilePath = br.readLine();		 // O/P file
			
			System.out.print("Tree Representation: ");
			treeNotation = br.readLine();

			/* Determine whether to terminate the program on the basis of 
				- difference in the values of original and new Q-matrices (Option 1)
				- number of iterations	(Option 2)
			*/	
            userOption = "1";
            
            if(userOption.charAt(0) == '1') {
                System.out.print("Q-value iteration cut-off value: ");
                minValue = Double.parseDouble(br.readLine());
                userSpecifiedIterationCount = 0;
            }
            else {
                System.out.print("Number of iterations: ");
                userSpecifiedIterationCount = Integer.parseInt(br.readLine());
                minValue = 0.0;
            }
		
		//System.out.print("Estimate invariant sites (Y/N): ");
		estimateInvariantSites = true;	
									
		} catch(Exception e) {
			System.out.println("Error occured while reading input parameters " 
								+ e);
		}
		
		iterateMaxAvgLikelihood(treeNotation, noOfSites, srcFilePath, destFilePath
					, provideQValFile.charAt(0), qValFilePath, minValue
                    , userSpecifiedIterationCount, userOption.charAt(0), false, estimateInvariantSites);
	}

	void iterateMaxAvgLikelihood(String treeNot, int siteNumber, String srcPath
				, String destPath, char qValFileOption, String qFilePath
				, double minVal, int uSpecIterationCount, char uOption
                		, boolean areSequencesRead, boolean estimateInvariantSites) {		

		int index = srcPath.lastIndexOf('/');

		try{
		
			FileWriter phylipFileWriter;
			FileWriter statsFileWriter = new FileWriter(srcPath.substring(0, index) + "/Stats.txt");			
			
			//Code added on 19-Sep-08
			//String treeNamesFile = srcPath.substring(0, index) + "/trees.txt";
			//BufferedReader treeFileReader = new BufferedReader(new FileReader(treeNamesFile));	
			
			for(int trees=0; trees<1; trees++) {

				String treeNotation = treeNot;

				TreeMap tPhyloTree = new TreeMap();	
		
				//Code added on 13-Nov-07
				t = new NewickTreeTraversal(false);
				t.generateTree(treeNotation, tPhyloTree);

				ArrayList tLeafNodes = t.determineLeafNodes(tPhyloTree);
				int taxaCount = tLeafNodes.size() + 1;

				String phylipFile = srcPath.substring(0, index) + "/Phylip.txt";

				double logL;
				FileReader srcFileReader = new FileReader(srcPath);
				BufferedReader srcFileBufRead = new BufferedReader(srcFileReader);						

				for(int i=0; i<noOfDataSets; i++) {
			
					System.out.println("Dataset = " + i);

					phylipFileWriter = new FileWriter(phylipFile);
					for(int j=0; j<taxaCount+1; j++) phylipFileWriter.write(srcFileBufRead.readLine() + "\n");
					srcFileBufRead.readLine(); //Read the blank line separating two data sets

					phylipFileWriter.close();

					logL = maxAvgLikelihood(treeNotation, siteNumber, phylipFile, destPath
								  , qValFileOption, qFilePath, minVal
            					        , uSpecIterationCount, uOption, false, i, estimateInvariantSites);

					statsFileWriter.write(logL + "\t");
					for(int x=0; x<4; x++) statsFileWriter.write(marginalProbVector[x] + "\t");

					statsFileWriter.write(beta + "\t");
					for(int x=0; x<4; x++) statsFileWriter.write(probGivenInv[x] + "\t");

					statsFileWriter.write("\n");

				} // All data sets considered

				srcFileReader.close();
			}// All tree topologies considered
			
			statsFileWriter.close();

		}catch(Exception e) {
			e.printStackTrace();
		}

	}

	
	/**	This method computes the maximum average likelihood based on Barry and 
	 *	Hartigan's algorithm
	 *
	 *	@param	treeNotation			Newick format tree
	 *	@param	siteNumber				Number of sites per species
	 *	@param	srcPath					Directory of nucleotide sequences
	 *	@param	destPath				Name and path of file used to store the
	 *									convergence Q-matrices
	 *	@param 	qValFileOption			1 -> Use Q-matrices based on Jukes-Cantor model 
	 *									2 -> Read Q-matrices from a user-sepcifed 
	 *										 file
	 *									3 -> Generate random Q-matrices
	 *	@param 	qFilePath				File name for reading initial values
	 *									of Q-matrices
	 *	@param 	minVal					Cut-off value. If the change in the 
	 *									elements of Q-matrices	< minValues, 
	 *									stop iterations
	 *	@param	uSpecIterationCount		User specified iteration count
	 *	@param 	uOption					1 -> Compute square of differences between
	 *										 matrices
	 *									2 -> Compute log likelihood a fixed number
	 *										 of times
	 *	@param	areSequencesRead		Determine whether the DNA sequences
	 *									have already been stored in an appropriate
	 *									data structure
	 *
	 *	@return Maximum average likelihood value for a user-sepcified tree
	 */
	
	double maxAvgLikelihood(String treeNotation, int siteNumber, String srcPath
				, String destPath, char qValFileOption, String qFilePath
				, double minVal, int uSpecIterationCount, char uOption
                , boolean areSequencesRead, int dataSetCount, boolean estimateInvariantSites) {		

		//re-set data structures that change from tree to tree
		initializeGlobalVariables();
	                
		//Added code to store the site-specific log likelihood values in a file
		//openSiteLogLikelihoodValueFile();
		
		//Assign input parameters to global variables
		srcFilePath = srcPath;
		destFilePath = destPath;
		qValFilePath = qFilePath;
		minValue = minVal;	
		noOfSites = siteNumber;	
						
		//Display the values for input parameters
		if(showOutput) {
			System.out.println("Src file path = " + srcFilePath
								+ " Dest file path = " + destFilePath
								+ " qValFilePath = " + qValFilePath
								+ " Min Value = " + minValue
								+ " no of sites = " + noOfSites
								+ " no of variable sites = " + noOfVariableSites);
		}
		
		double logLikelihoodValue = 0.0;
		
		//TreeTraversal t = new TreeTraversal();  // class for creating the tree
		NodeStructure ns = new NodeStructure(); //class for storing nodes
		
		boolean foundTopIntNode; //determines formula to use for Internal nodes
		
		System.out.println("Program Started -> " +  new Date());
		
		//Initialize array for storing (site, parent node, child node values)
		siteSpecificParentChildNodeVal = new double[noOfSites][4][4];
		
		//Create the tree based on user-specified Newick tree representation
		//Code added on 13-Nov-07
		t = new NewickTreeTraversal(false);
		t.generateTree(treeNotation, phylogeneticTree);
				
		String rootNode = t.getRootNode(phylogeneticTree);
		
		if(showOutput) {
			System.out.println("Created Phylogenetic tree -> " +  new Date());	
		}
				
		/* Determine all leaf nodes. Read sequences from various files based
		 * on the order in which leaf nodes appear in the Newick tree 
		 * representation from left to right
		 */
		//leafNodesList = t.determineLeafNodes();	
		leafNodesList = t.determineLeafNodes(phylogeneticTree);	
					
		//Assume that the sequences are character sequences i.e {A,C,G,T}
		if(!areSequencesRead) {
			readSequences("A", leafNodesList, rootNode);
			System.out.println("Sequences read");	
		}		

		if(showOutput) {
			System.out.println("Completed reading of sequences -> " +  new Date());
		}	
				
		/* Determine all internal nodes. Initialize Q-values along the branches
		 * specified by the internal nodes
		 */
		intNodesList = t.determineInternalNodes(phylogeneticTree);
		
		switch(qValFileOption) {
			case '1': initializeProbabilities(intNodesList, leafNodesList);
					  break;	
			case '2': initializeProbabilities(intNodesList, leafNodesList, 'Y');
					  break;	
			case '3': generateRootQMatrix(rootNode);
					  //Store the initial Q-values in a file
	  				  generateInitQMatricesFile(intNodesList, leafNodesList
	  				  							, "InitMatrices.txt");
		}
		
		if(showOutput) {
			System.out.println("Probability distribution values initialized " 
								+  " -> " + new Date());
		}						
		
		//Display phylogenetic tree
		if(showOutput) {
			//t.displayTree(phylogeneticTree);
			t.displayTree();
			System.out.println(" ");
			System.out.println(" ");
		}
				
		//Open file for storing Q-values and Log Likelihood values								
		openFile();
		
		//Initialize TreeMap for RiXi
		probRiXi = new TreeMap[noOfVariableSites]; 
		probSiXi = new TreeMap [noOfVariableSites];
		
		//Initialize branch update option
		initializeUpdateOption(intNodesList, leafNodesList, noOfSites);
		
		/*	Logic for SBH+I model
		 *	Step 1: Assume that 25% of unvaried sites are invariant
		 *	Step 2: Remove the invariant sites and estimate Q-matrices and Pi-vector
		 *	Step 3: Use the current estimates of Q-matrices to obtain new estimates 
		 *		  of invariant sites
		 *	Step 4: Go to Step 2
		 */	
		
		int[] invSitesCount = new int[4];
		int[] origUnvariedSitesCount = new int[4];
		int sumUnvariedSites = 0;
	
		double[] origMargProbVector = new double[4];	
		double sumUnvariedSitesProb = 0.0;

		for(int inv=0; inv<4; inv++) {
			invSitesCount[inv] = (int) Math.round(0.25 * patternCountArray[unvariedSitesIndex[inv]]);
			origUnvariedSitesCount[inv] = patternCountArray[unvariedSitesIndex[inv]];
			
			origMargProbVector[inv] = marginalProbVector[inv];

			sumUnvariedSitesProb += origUnvariedSitesCount[inv]*1.0/ noOfSites;
			sumUnvariedSites += origUnvariedSitesCount[inv];
		}

		System.out.println("Unvaried Sites Count = " + sumUnvariedSites); 
			
		//Default values for beta and P(X|inv)
		beta = 0.25 * sumUnvariedSitesProb;
		for(int inv=0; inv<4; inv++) probGivenInv[inv] = invSitesCount[inv]/
										(invSitesCount[0] + invSitesCount[1] + invSitesCount[2] + invSitesCount[3]);

		double partialLogL = 0.0;
		double totalLogL = 0;
		double marginalProbSum;

		int revisedSitesCount;
		int piCount;
		int betaCount = 0;

		//while(betaCount++ < 50) {
		while(betaCount++ < 20) { 

			//Obtain the marginal probability of variable sites
			marginalProbSum = 0;

			for(int inv=0; inv<4; inv++) {
				marginalProbVector[inv] = origMargProbVector[inv] * noOfSites - invSitesCount[inv];
				marginalProbSum += marginalProbVector[inv];
			}

			for(int inv=0; inv<4; inv++) marginalProbVector[inv] /= marginalProbSum; 

			//Obtain the actual no of sites to be considered
			revisedSitesCount = noOfSites - (invSitesCount[0] 
									+ invSitesCount[1] 
									+ invSitesCount[2] 
									+ invSitesCount[3]); 

			System.out.println("Revised no of sites = " + revisedSitesCount);

			//Obtain the number of variable sites of type {AA...A}
			for(int inv=0; inv<4; inv++) 
				patternCountArray[unvariedSitesIndex[inv]] =  origUnvariedSitesCount[inv] - invSitesCount[inv];
		
			//Parameter estimation for variable sites
			piCount = 0;

			while(piCount++ < 2) {

				//for(int a=0; a<4; a++) System.out.println(modifiedMargProb[a] + ", " + marginalProbVector[a]);

				//Initialize branch update option
				initializeUpdateOption(intNodesList, leafNodesList, revisedSitesCount);

				//Initialize probabilities
				initializeProbabilities(intNodesList, leafNodesList);

				partialLogL = estimateQMatrices(rootNode);
				//System.out.println("Partial Log L = " + partialLogL);

				//Update marginal probability vector
				updateMargProbVector();	
			}

			//Obtain total log likelihood
			totalLogL = partialLogL;

			for(int inv=0; inv<4; inv++) 
				totalLogL -= patternCountArray[unvariedSitesIndex[inv]] * Math.log(unvariedSitesProb[inv]);

			totalLogL = totalLogL + (noOfSites - sumUnvariedSites) * Math.log(1-beta);

			for(int inv=0; inv<4; inv++) 
				totalLogL += origUnvariedSitesCount[inv] * Math.log((1-beta) * unvariedSitesProb[inv]  + beta * probGivenInv[inv]);

			System.out.println("Total log L = " + totalLogL);

			//Obtain invariant sites parameters			
			double sumVariedSitesProb = 0.0;

			for(int inv=0; inv<4; inv++) sumVariedSitesProb += unvariedSitesProb[inv];

			beta = (sumUnvariedSitesProb - sumVariedSitesProb)/ (1 - sumVariedSitesProb);
			
			for(int inv=0; inv<4; inv++) {
				probGivenInv[inv] = ((origUnvariedSitesCount[inv]*1.0/ noOfSites) - unvariedSitesProb[inv] * (1-beta))/ beta;

				invSitesCount[inv] = (int) Math.round(noOfSites * beta * probGivenInv[inv]);
			}	
			
		}//All parameters estimated

		
		/* Obtain the final Q-matrices for all internal and leaf nodes. Used
		 * for easier comparison of Q-matrix values generated under different 
		 * conditions
		 */
	    writeTextToFile("");
	    writeTextToFile("******************************************");
	    writeTextToFile("************ FINAL Q-MATRICES ************");
	    writeTextToFile("******************************************");
                
		writeFinalQMatrices(branchList); 
		writeQMatFile(branchList, dataSetCount);

		//Obtain the log likelihoods with the Q-values at convergence
	    writeTextToFile("");
	    writeTextToFile("******************************************");
	    writeTextToFile("************* FINAL L VALUES *************");
	    writeTextToFile("******************************************");

		logLikelihoodValue = iterateLogLikelihoodAndQValues(rootNode, 2, uOption);
	
		//close the output file
		closeFile();
		
		System.out.println("Program Completed -> " +  new Date());

		//return logLikelihoodValue;
		return totalLogL;	
	}
	
	/**
	 * This method lists the DNA sequence files to be read
	 * The sequences could be of the form {A,C,G,T} or {0,1,2,3}
	 *
	 * @param seqChar 	 A -> alphabets i.e. {A|a C|c G|g T|t}
	 *					 N -> numbers i.e. {0 1 2 3}
	 * @param leafNodesList	Read relevant sequence files
	 */
	void readSequences(String seqChar, ArrayList leafNodes, String root) {		
						
		if(seqChar.equals("A")) {
			readAllSequences(leafNodes, root);			
		}
		else {
			//Add code for handling numbers instead of characters
		}
	}	

	/**
	 * This method reads DNA sequences for all taxa and identifies all unique
	 * patterns. A pattern is stored in the linked lists of sequences only if
	 * it occurs for the first time. A HashMap stores the number of times 
	 * a pattern occurs
	 *
	 * @param leafNodes		List of leaf nodes excluding root node
	 * @param root			Root node name
	 */
	void readAllSequences(ArrayList leafNodes, String root) {
		
		int i;		
		int size = leafNodes.size();
		int patternSite;
		int patternCount;		
		
		//No of sites with atleast 1 sequence having '-' pattern
		int sitesToBeDeleted = 0;	

		char nucleotide[] = new char[size + 1];	

		HashMap patternMap = new HashMap();
		
		//Store input sequences 
		String inputSequences[] = new String[size + 1];		
		String pattern;
		String dnaSequence;

		BufferedReader br;
		ArrayList ls[] = new ArrayList[size + 1];
		ArrayList patternCountList = new ArrayList();
		
		double[] avgProb = new double[4];
		for(int j=0; j<4; j++) avgProb[j] = 0;
		
		//Initialize number of variable sites
		noOfVariableSites = 0;

		//Determine the strings corresponding to unvaried sites
		String[] unvariedSitesPattern = new String[4];

		int unvariedPatternLength = leafNodes.size() + 1;

		for(i=0; i<4; i++) unvariedSitesPattern[i] = "";

		for(i=0; i<unvariedPatternLength; i++) {
			unvariedSitesPattern[0] += "A";
			unvariedSitesPattern[1] += "C";
			unvariedSitesPattern[2] += "G";
			unvariedSitesPattern[3] += "T";
		}
		
        try
        {
			//br = new BufferedReader(new FileReader(srcFilePath + sequenceFileName));
			FileReader srcFileReader = new FileReader(srcFilePath);
			br = new BufferedReader(srcFileReader);
			
			// Ignore the first line of PHYLIP file
			br.readLine();
			
			for(i=0; i<=size; i++) {
				dnaSequence = br.readLine();
				
				String speciesName = dnaSequence.substring(0,10);
				speciesName = speciesName.trim() + "-Node";
				
				String speciesSeq = "";
				
				//Read the sequence for a given species
				if(dnaSequence.length() > 10) 
					speciesSeq = dnaSequence.substring(10);
					
				int noOfBasesRead = speciesSeq.length();
				
				/*
				System.out.println("No of bases read = " + noOfBasesRead
								+ ", DNA Sequence length = " + dnaSequence.length());
				*/
				
				while(noOfBasesRead < noOfSites) {
					speciesSeq += br.readLine();
					noOfBasesRead = speciesSeq.length();
				}					
				
				// Code Added on 09-Jan-06
				avgProb = getMarginalProb(speciesSeq, avgProb, noOfSites, i);
				 //New Code ends
				
				/**	Add species name to the array of sequence names
				 *	The root node is assigned the highest index in the array
				 *	A leaf node's index is determined by the its order in the 
				 *	ArrayList of leaf nodes	
				 */
				if(speciesName.equals(root)) {
					//inputSequences[size] = dnaSequence.substring(10);
					inputSequences[size] = speciesSeq;
				}
				else {
					int index = leafNodes.indexOf(speciesName);
					//inputSequences[index] = dnaSequence.substring(10);
					inputSequences[index] = speciesSeq;
				}
			}
			
			//System.out.println("Pattern identification started");
			
			// Obtain the average Pi-vector 
			for(int j=0; j<4; j++) {
				marginalProbVector[j] = avgProb[j] * 1.0/ (size+1);
			}
						
			//Initialize arraylist
			for(i=0; i<=size; i++) {
				ls[i] = new ArrayList();
			}

			//identify patterns
            for(int j=0; j<noOfSites; j++) {
            	pattern = "";
            	
            	// Determine whether '-' character encountered in any sequence 
            	// for a particular site
            	boolean isSiteNotAssigned = false;  
                
            	for(i=0; i<=size; i++) {
            		nucleotide[i] = inputSequences[i].charAt(j);
                    
                    //delete a site which has one or more '-'
                    if((nucleotide[i] == '-') && (!isSiteNotAssigned)){
                    	isSiteNotAssigned = true;
                    	sitesToBeDeleted++;
                    }	
                    
            		pattern += nucleotide[i];
            	}
            	
            	//determine if the pattern already exists          	
            	if(patternMap.containsKey(pattern) && !isSiteNotAssigned) {
            		patternSite = ((Integer)patternMap.get(pattern)).intValue();
            		patternCount = 
            			((Integer)patternCountList.get(patternSite)).intValue()
            			 + 1;
            		
            		patternCountList.set(patternSite, new Integer(patternCount));
            	}
            	else if(!isSiteNotAssigned) {            		

				//Code added -> 23-Aug-06
				if(pattern.equals(unvariedSitesPattern[0])) unvariedSitesIndex[0] = noOfVariableSites;
				if(pattern.equals(unvariedSitesPattern[1])) unvariedSitesIndex[1] = noOfVariableSites;
				if(pattern.equals(unvariedSitesPattern[2])) unvariedSitesIndex[2] = noOfVariableSites;
				if(pattern.equals(unvariedSitesPattern[3])) unvariedSitesIndex[3] = noOfVariableSites;
				//Code ends -> 23-Aug-06

            		patternMap.put(pattern, new Integer(noOfVariableSites));            		
            		patternCountList.add(new Integer(1));
            		noOfVariableSites++;
            		
            		for(i=0; i<=size; i++) {
                            switch(nucleotide[i]) {
	            				case 'A': ls[i].add(new Integer(0)); break;
	            				case 'a': ls[i].add(new Integer(0)); break;
	            				case 'C': ls[i].add(new Integer(1)); break;
	            				case 'c': ls[i].add(new Integer(1)); break;
	            				case 'G': ls[i].add(new Integer(2)); break;
	            				case 'g': ls[i].add(new Integer(2)); break;
	            				case 'T': ls[i].add(new Integer(3)); break;
	            				case 't': ls[i].add(new Integer(3)); 
	            			}
            		}
            		
            	} //End if-then-else for pattern            	
           	
            } // end FOR loop for no of sites
            
            //Store the patterns for all leaf nodes           
            for(i=0; i<size; i++) {
				treeSequences.put((String)leafNodes.get(i), ls[i]);
				ls[i] = null; //release memory
            }	
            
            //Store the patterns for root node
            treeSequences.put(root, ls[size]);
            ls[size] = null;
			
			//Convert Linked List into array for faster access
			int listSize = patternCountList.size();
			
			patternCountArray = new int[listSize];
			
			for(i=0; i<listSize; i++) {
				patternCountArray[i] = ((Integer)patternCountList.get(i)).intValue();
			}

            //ls[size] = null;                 
     		patternMap = null;     			  
     		patternCountList = null;
     		
     		//Delete the sites having '-' pattern from the actual site count
     		noOfSites -= sitesToBeDeleted;
     		
     		System.out.println("Modified number of sites = " + noOfSites
					+ " no of variable sites = " + noOfVariableSites);
     		
     		srcFileReader.close();			
        }        
        catch (Exception e)
        {
            System.out.println("Error occured while reading files " 
            					+ " Exception = " + e);
        }
        
	}
	
	/**
	 * This method initializes the joint probability distribution values along
	 * all branches specified by internal nodes and leaf nodes
	 *
	 * @param intNodesList		List of internal nodes
	 * @param leafNodesList		List of leaf nodes
	 * @param initialize		Value set to Y (Used for method overloading)
	 */	
	void initializeProbabilities(ArrayList intNodesList
								, ArrayList leafNodesList, char initialize) {
		BranchDetails bd;
		double qValArray[][] = new double[4][4];
		int i,j,k;
		String rowValues;
		StringTokenizer str;
		
		FileReader fr;
		
		try{
			fr = new FileReader(qValFilePath);
			BufferedReader br = new BufferedReader(fr);
			
			//Initialize for all internal nodes		
			for(i=0; i<intNodesList.size(); i++) {						
				//System.out.println("Reading for internal node: " 
				//					+ intNodesList.get(i));
				
				for(j=0; j<4; j++) {
					//Read a row of Q-matrix and identify A, C, G, T values
					rowValues = br.readLine();
					str = new StringTokenizer(rowValues, "\t");

					for(k=0; k<4; k++) {						
						qValArray[j][k] = Double.parseDouble((str.nextToken()));
					}
				}
												
				bd = new BranchDetails();			
				bd.setJointProbability(qValArray);
				
				branchList.put((String)intNodesList.get(i), bd);
				
				//Read the blankline separating two Q-matrices
				rowValues = br.readLine();
			}
			
			//Initialize for all leaf nodes
			for(i=0; i<leafNodesList.size(); i++) {
				//System.out.println("Reading for leaf node: " 
				//						+ leafNodesList.get(i));

				for(j=0; j<4; j++) {
					//Read a row of Q-matrix and identify A, C, G, T values
					rowValues = br.readLine();
					str = new StringTokenizer(rowValues, "\t");

					for(k=0; k<4; k++) {
							qValArray[j][k] = Double.parseDouble((str.nextToken()));
					}
				}
	
				bd = new BranchDetails();
				bd.setJointProbability(qValArray);
				
				branchList.put((String)leafNodesList.get(i), bd);
				rowValues = br.readLine();
			}
			
			fr.close();
			
		} catch(Exception e) {
			System.out.println("Error Occured while initializing Q-values "
								+ e);				
		}		
				
	}
	
	/**
	 * This method initializes the joint probability distribution values along
	 * all branches specified by internal nodes and leaf nodes
	 *
	 * @param intNodesList		List of internal nodes
	 * @param leafNodesList		List of leaf nodes
	 */	
	
	void initializeProbabilities(ArrayList intNodesList
									, ArrayList leafNodesList) {
		BranchDetails bd;
		int i;
		
		//Initialize for all internal nodes		
		for(i=0; i<intNodesList.size(); i++) {
			bd = new BranchDetails();
			bd.setJointProbability();
			branchList.put((String)intNodesList.get(i), bd);
		}
		
		//Initialize for all leaf nodes
		for(i=0; i<leafNodesList.size(); i++) {
			bd = new BranchDetails();
			bd.setJointProbability();
			branchList.put((String)leafNodesList.get(i), bd);
		}	
	}
	
	//This method opens a file for storing log likelihood and Q-values
	void openFile() {
		try{
			fw = new FileWriter(destFilePath);
		} catch(Exception e) {
			System.out.println("Error opening output file " + e);
		}	
	}
	
	// This method closes the file
	void closeFile() {
		try {
			fw.close();
		} catch (Exception e) {
			System.out.println("Error occured while closing output file " + e);
		}
		
	}
	
        void writeTextToFile(String str) {
		try{
			fw.write(str + "\n");		
		} catch(Exception e) {
			System.out.println("Error occured while entering text ");
		}            
        }
        
	/**
	 * This method is used to store Log Likelihood value along a node in the
	 * output file
	 *
	 * @param node	Tree node at which log likelihood value was computed
	 * @param value	Computed value of log likelihood
	 */
	void writeOutput(String node, double value) {
		String data = "\n\n Log Likelihood value at node " + node 
						+ " = " + value + "\n"; 			
		try{
			fw.write(data);		
		} catch(Exception e) {
			System.out.println("Error occured while entering log " 
								+ " likelihood value ");
		}
	}

	/**
	 * This method is used to store iteration number in the output file
	 *
	 * @param itrNumber		Iteration Number
	 */
	void writeOutput(int itrNumber) {
		String data = "\n\n Iteration Number " + itrNumber + "\n"; 			
		try{
			fw.write(data);		
		} catch(Exception e) {
			System.out.println("Error occured while entering iteration no");
		}
	}

	/**
	 * This method is used to store Q-values for a particular branch in the 
	 * output file
	 *
	 * @param node			Branch of phylogenetic tree
	 */
	void writeOutput(String node) {
		BranchDetails bd; 
		String data;
		double[][] probDistribution;
				
		bd = (BranchDetails)updateBranchList.get(node);
		probDistribution = bd.getJointProbability();

		try {
			fw.write("\n");
			fw.write("Probability Distribution Values b/w parent node "
					+ " and child node (" + node + ") \n" );
		} catch (Exception e) {
			System.out.println("Error occured while writing a newline "
								+ " character " + e);
		}
		
		for(int i=0; i<4; i++) {
			for(int j=0; j<4; j++) {
				data = probDistribution[i][j] + "\t";
				
				try {
					fw.write(data);
				} catch (Exception e) {
					System.out.println("Error occured while writing prob "
										+ " distribution values " + e);
				}	
			}  // end of inner FOR loop
			
			//add a new line character each time the i-value changes
			try {
				fw.write("\n");
			} catch (Exception e) {
				System.out.println("Error occured while writing a newline "
									+ " character " + e);
			}
			
		} // end of outer FOR loop				
	}	
	
	/**
	 * This method is used to determine the log likelihood value and updation
	 * of Q-values for the internal node that satisfies the condition T(j)=0
	 * The phylogenetic tree is traversed down from the internal node 
	 * to all the leaf nodes (excluding the root node)
	 *
	 * 	@param 	nodeName			Name of internal node
	 * 	@param 	rootNode			Name of root node
	 * 	@param 	option				1 -> Compute log likelihood and update Q-matrices
	 *								2 -> Only compute log likelihood values	
	 *	@param	iterationOption		1 -> Compute square of differences between
	 *									 original and new Q-matrices
	 *								2 -> Do not compute
	 *
	 *	@return			 			logLikelihood value
	 */
	double computeValuesRootToInt(String nodeName, String rootNode, int option
									, char iterationOption) {
		ArrayList rootNodeSequence; 
		ArrayList probDist = new ArrayList();
		BranchDetails bdUpdated;
		BranchDetails bd;		
		InternalNodeRiXi nRiXi; 
		double[][] jointProbDistribution;
		double logLikelihood =0.0;
		double partialLogLikelihoood = 0.0;
		double siteProb;
		double sum;
		double denominator; //store the denominator value for Q-updation
		int base;
		int i;
		int j;
		int k;		

		//Obtain the sequence for root node
		rootNodeSequence = (ArrayList)treeSequences.get(rootNode);		
		
		//Obtain the joint probability distribution along root -> internal_node
		bd = (BranchDetails)branchList.get(nodeName);
		
		jointProbDistribution = bd.getJointProbability();
		
		for(i=0; i<noOfVariableSites; i++) {					
			siteProb = 0.0;
			
			//obtain the base at site i
			base = ((Integer)rootNodeSequence.get(i)).intValue();			
			nRiXi = new InternalNodeRiXi();
				
			nRiXi.setNucleotideA(pRiX1(nodeName, 0, i)) ;
			nRiXi.setNucleotideC(pRiX1(nodeName, 1, i)) ;
			nRiXi.setNucleotideG(pRiX1(nodeName, 2, i)) ;
			nRiXi.setNucleotideT(pRiX1(nodeName, 3, i)) ;
				
			probRiXi[i].put(nodeName, nRiXi);
		
			baseProb = nRiXi.getNucleotides();
			siteSpecificParentChildNodeVal[i][0][0] = baseProb[0];
			siteSpecificParentChildNodeVal[i][1][0] = baseProb[1];
			siteSpecificParentChildNodeVal[i][2][0] = baseProb[2];
			siteSpecificParentChildNodeVal[i][3][0] = baseProb[3];
						
			/* For each site, consider all 4 values i.e. {A,C,G,T}
			 * Q(Xoi,Xji)P(G2j|Xji) sum over all Xji
			 * G2j represents the leaf nodes connected to node j
			 */			
			siteProb = jointProbDistribution[base][0] 
					* siteSpecificParentChildNodeVal[i][0][0];
			
			siteProb += jointProbDistribution[base][1] 
					* siteSpecificParentChildNodeVal[i][1][0];
			
			siteProb += jointProbDistribution[base][2] 
					* siteSpecificParentChildNodeVal[i][2][0];
					
			siteProb += jointProbDistribution[base][3] 
					* siteSpecificParentChildNodeVal[i][3][0];

			probDist.add(new Double(siteProb));						
			logLikelihood += patternCountArray[i] * Math.log(siteProb); //Uncommented on 13-Nov-07

			//Code added -> 23-Aug-06
			if(i == unvariedSitesIndex[0]) unvariedSitesProb[0] = siteProb;
			if(i == unvariedSitesIndex[1]) unvariedSitesProb[1] = siteProb; 
			if(i == unvariedSitesIndex[2]) unvariedSitesProb[2] = siteProb;
			if(i == unvariedSitesIndex[3]) unvariedSitesProb[3] = siteProb; 
			//Code ends -> 23-Aug-06			

			
		}
				
		//Store the value of log likelihood in a file along with the node name
		writeOutput(nodeName, logLikelihood);

		if(option == 1) {
			/* Update joint probability distribution along root -> internal_node
			 * i represents root node
			 * j represents internal node
			 */
			for(i=0; i<4; i++) {
	
				//for each value of X0, consider all 4 values of X1
				for(j=0; j<4; j++) {
					
					denominator = 0.0;
					sum = 0.0;
					
					/* For a particular (i,j) value, consider only those sites
					 * which have the same base as i value
					 */				
					for(k=0; k<noOfVariableSites; k++) {
						base = ((Integer)rootNodeSequence.get(k)).intValue();
						
						if(base == i) {
							siteProb = jointProbDistribution[i][j] 
								* siteSpecificParentChildNodeVal[k][j][0];
							
							//Q(Xo,X1)P(G2j|X1) sum over all values of X1			
							denominator = ((Double)probDist.get(k)).doubleValue();
							
							sum += patternCountArray[k] * siteProb/denominator;
																						
						} // consider only those sites for which X(k) = X(i)
										
					} // end of FOR loop for no_of_sites
									
					//divide by N (= no_of_sites)
					//sum = sum/ noOfSites;
					
					//set new probability value for (i,j)
					updateMatrix[i][j] = sum;			
					
				} //end of FOR loop for child node
			} //end of FOR loop for parent node

			/*	The matrix updateMatrix[i][j] contains values that have not been divided
			 *	by the Lagrange multiplier. Using the existing values of the
			 *	Lagrange multipliers, new values are obtained. These new values
			 *	are used to obtain revised estimates of the Q-matrices
			 */
			
			LagrangeMultipliers lagrangeEstimates = (LagrangeMultipliers) updateLagMultPerBranch.get(nodeName);

			double[] currentLambdaEst = lagrangeEstimates.getRowMultipliers();
			double[] currentDeltaEst = lagrangeEstimates.getColMultipliers();

			double[] updatedLambdaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 1);
			double[] updatedDeltaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 2);
		
			lagrangeEstimates.setRowMultipliers(updatedLambdaEst);
			lagrangeEstimates.setColMultipliers(updatedDeltaEst);
			updateLagMultPerBranch.put(nodeName, lagrangeEstimates);

			//Get the revised estimates of Q-matrix
			for(int parent=0; parent<4; parent++) {
				for(int child=0; child<4; child++) {
					updateMatrix[parent][child] /= (updatedLambdaEst[parent] + updatedDeltaEst[child]);
				}
			}

			//Release memory			
			currentLambdaEst = null;
			currentDeltaEst = null;

			updatedLambdaEst = null;
			updatedDeltaEst = null;

			probDist = null;

			//Obtain new values for the branch			
			bdUpdated = new BranchDetails();
			bdUpdated.setJointProbability(updateMatrix);
					
			/* Updated Q-matrix values are store in a temporary TreeMap that changes
			 * with every iteration. Those branches where original and re-calculated
			 * Q-matrix values are different (as determined by the function 
			 * setUpdateOption), the branchList TreeMap is updated with the new
			 * Q-matrix values  
			 */
			updateBranchList.put(nodeName, bdUpdated);
	            if(iterationOption != '2') setUpdateOption(nodeName, bd, bdUpdated);
				
			//updateMatrix = null;	//release the resource
			bd = null;
			bdUpdated = null;
			
			//Store the updated Q-values in the file 
			writeOutput(nodeName);
			
		} //end of OPTION 
				
		return logLikelihood;
	}		

	/**
	 * This method is used to determine the log likelihood value and updation
	 * of Q-values for leaf nodes. The phylogenetic tree is traversed 
	 * from a particular leaf node to all other end nodes
	 *
	 * @param nodeName				Name of internal node
	 * @param leafNode				Name of leaf node
	 * @param option				1 -> Compute log likelihood and update Q-matrices
	 *								2 -> Only compute log likelihood values	
	 * @param iterationOption		1 -> Compute square of differences between
	 *									 original and new Q-matrices
	 *								2 -> Do not compute
	 *
	 *	@return 					Log Likelihood value
	 */
	double computeValuesIntToLeaf(String nodeName, String leafNode, int option,
									 char iterationOption) {
		ArrayList leafNodeSequence; 
		ArrayList probDist = new ArrayList();
		BranchDetails bd;		
		BranchDetails bdUpdated;
		double[][] jointProbDistribution;
		double logLikelihood =0.0;
		double siteProb;
		double sum;
		double prob;
		double marginalProb;
		double denominator; //store the denominator value for Q-updation
		int base;
		int i;
		int j;
		int k;
		
		//Obtain the sequence for leaf node
		leafNodeSequence = (ArrayList)treeSequences.get(leafNode);
		
		//Obtain the joint probability distribution along (leaf, internal_node)
		bd = (BranchDetails)branchList.get(leafNode);
		jointProbDistribution = bd.getJointProbability();
		
		BranchDetails bd1 = (BranchDetails)branchList.get(nodeName);
		double[][] jP1 = bd1.getJointProbability();
		
		for(i=0; i<noOfVariableSites; i++) {
			siteProb = 0.0;
			
			//obtain the base at site i
			base = ((Integer)leafNodeSequence.get(i)).intValue();
			
			/* For each site, consider all 4 values i.e. {A,C,G,T}
			 * Q(Xki,Xji)P(G1j|Xji) sum over all Xji
			 * G1j represents all end nodes connected to node j excluding
			 * the current leaf node
			 */					
			 
			//[0][base] is used as the direction of traversal is child -> parent 
			prob = pSiX2(nodeName, leafNode, 0, i);						
			siteSpecificParentChildNodeVal[i][0][0] = prob;
			
			marginalProb = jP1[0][0] + jP1[1][0] + jP1[2][0] + jP1[3][0];
			
			siteProb = jointProbDistribution[0][base] 
						* prob/ marginalProb;
						
			prob = pSiX2(nodeName, leafNode, 1, i);			
			siteSpecificParentChildNodeVal[i][1][0] = prob;
			
			marginalProb = jP1[0][1] + jP1[1][1] + jP1[2][1] + jP1[3][1];
			
			siteProb += jointProbDistribution[1][base] 
						* prob/marginalProb;
			
			prob = pSiX2(nodeName, leafNode, 2, i);
			siteSpecificParentChildNodeVal[i][2][0] = prob;
			
			marginalProb = jP1[0][2] + jP1[1][2] + jP1[2][2] + jP1[3][2];
									
			siteProb += jointProbDistribution[2][base]
						* prob/marginalProb;
			
			prob = pSiX2(nodeName, leafNode, 3, i);
			siteSpecificParentChildNodeVal[i][3][0] = prob;
			
			marginalProb = jP1[0][3] + jP1[1][3] + jP1[2][3] + jP1[3][3];
						
			siteProb += jointProbDistribution[3][base] 
						*prob/marginalProb;
			
			probDist.add(new Double(siteProb));

			logLikelihood += patternCountArray[i] * Math.log(siteProb);
		}
		
		//Store the value of log likelihood in a file along with the node name
		writeOutput(leafNode, logLikelihood);

		if(option == 1) {
			/* Update joint probability distribution along leaf -> internal_node
			 * i represents internal node
			 * j represents leaf node
			 */
			for(i=0; i<4; i++) {
	
				//for each value of X0, consider all 4 values of X1
				for(j=0; j<4; j++) {
					
					denominator = 0.0;
					sum = 0.0;
					
					/* For a particular (i,j) value, consider only those sites
					 * which have the same base as i value
					 */				
					for(k=0; k<noOfVariableSites; k++) {
						base = ((Integer)leafNodeSequence.get(k)).intValue();
						
						if(base == j) {
							marginalProb = jP1[0][i] + jP1[1][i] + jP1[2][i] 
											+ jP1[3][i];			
											
							siteProb = jointProbDistribution[i][j]
										* siteSpecificParentChildNodeVal[k][i][0]
										/ marginalProb;
							
							denominator = ((Double)probDist.get(k)).doubleValue();
	
							sum += patternCountArray[k] * siteProb/denominator;
	
						} // consider only those sites for which X(k) = X(i)
										
					} // end of FOR loop for no_of_sites
													
					//divide by N (= no_of_sites)
					//sum = sum/ noOfSites;
					
					//set new probability value for (i,j)
					updateMatrix[i][j] = sum;				
									
				} //end of FOR loop for leaf node
			} //end of FOR loop for internal node
			
			LagrangeMultipliers lagrangeEstimates = (LagrangeMultipliers) updateLagMultPerBranch.get(nodeName);

			double[] currentLambdaEst = lagrangeEstimates.getRowMultipliers();
			double[] currentDeltaEst = lagrangeEstimates.getColMultipliers();

			double[] updatedLambdaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 1);
			double[] updatedDeltaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 2);
		
			lagrangeEstimates.setRowMultipliers(updatedLambdaEst);
			lagrangeEstimates.setColMultipliers(updatedDeltaEst);
			updateLagMultPerBranch.put(nodeName, lagrangeEstimates);

			//Get the revised estimates of Q-matrix
			for(int parent=0; parent<4; parent++) {
				for(int child=0; child<4; child++) {
					updateMatrix[parent][child] /= (updatedLambdaEst[parent] + updatedDeltaEst[child]);
				}
			}

			//Release memory			
			currentLambdaEst = null;
			currentDeltaEst = null;

			updatedLambdaEst = null;
			updatedDeltaEst = null;
					
			probDist = null;
			
			bdUpdated = new BranchDetails();		
			bdUpdated.setJointProbability(updateMatrix);
			
			//store the new joint probability distribution values
			//branchList.put(leafNode, bd);
			
			/* Updated Q-matrix values are store in a temporary TreeMap that changes
			 * with every iteration. Those branches where original and re-calculated
			 * Q-matrix values are different (as determined by the function 
			 * setUpdateOption), the branchList TreeMap is updated with the new
			 * Q-matrix values  
			 */
			updateBranchList.put(leafNode, bdUpdated);
            if(iterationOption != '2') setUpdateOption(leafNode, bd, bdUpdated);
			
			//setUpdateOption(leafNode, bd, bdUpdated);
	
			bd = null;		
			bdUpdated = null;
			
			//Store the updated Q-values in the file 
			writeOutput(leafNode);
		
		} // end of OPTION
		
		return logLikelihood;
	}	

	/**
	 * This method is used to determine the log likelihood value and updation
	 * of Q-values for inernal nodes which satisfy the condition T(j) <> 0
	 *
	 * @param nodeName				Name of internal node
	 * @param parentNode			Name of internal node's parent node
	 * @param option				1 -> Compute log likelihood and update Q-matrices
	 *								2 -> Only compute log likelihood values	
	 * @param iterationOption		1 -> Compute square of differences between
	 *									 original and new Q-matrices
	 *								2 -> Do not compute
	 *
	 *	@return			 			Log Likelihood value
	 */
	double computeValuesIntToInt(String nodeName, String parentNode, int option,
									 char iterationOption, boolean computeMargProb) {

		BranchDetails bd;
		BranchDetails bdUpdated;
		ArrayList probList = new ArrayList();
		double[][] jointProbDistribution;
		double logLikelihood =0.0;
		double siteProb;
		double sum;
		double denominator; //store the denominator value for Q-updation		
		int base;
		int i;
		int j;
		int k;
		
		//Obtain the joint probability distribution along (int_node, parent)
		bd = (BranchDetails)branchList.get(nodeName);
		jointProbDistribution = bd.getJointProbability();					

		double[] likelihoodPerBase = new double[4];
		double margProbSum;

		/* Compute log likelihood for variable sites
		 * i represents the no of sites
		 * k represents {A,C,G,T} for parent
		 * j represents {A,C,G,T} for internal node
		 */
		for(i=0; i<noOfVariableSites; i++) {
			siteProb = 0.0;
			
			for(j=0; j<4; j++) {	// base at parent node
			
				likelihoodPerBase[j] = 0;

				for(k=0; k<4; k++) {// base at child node 
					//direction of traversal is from parent -> internal node			
					//(site, parent node, child node)
					siteSpecificParentChildNodeVal[i][j][k] 
								= pG1jG2j(nodeName, parentNode, k, j, i);
					
					siteProb += jointProbDistribution[j][k] 
								* siteSpecificParentChildNodeVal[i][j][k]; 

					//Compute P(base) * P(leaf_nodes|base)
					likelihoodPerBase[j] += jointProbDistribution[j][k] 
								* siteSpecificParentChildNodeVal[i][j][k]; 
							
				} //end of FOR loop for parent node
				
			} //end of FOR loop for internal node

			//Estimate the new marginal probability vector
				margProbSum = likelihoodPerBase[0] + likelihoodPerBase[1] 
							+ likelihoodPerBase[2] + likelihoodPerBase[3];

				for(int margProbBase=0; margProbBase<4; margProbBase++) 
					modifiedMargProb[margProbBase] += patternCountArray[i] 
											* likelihoodPerBase[margProbBase] 
											/ margProbSum;
				

			//Marginal probability estimation completed 
			
			probList.add(new Double(siteProb));

			logLikelihood += patternCountArray[i] * Math.log(siteProb);
							
		}
		
		//Store the value of log likelihood in a file along with the node name
		writeOutput(nodeName, logLikelihood);
	
		if(option ==1) {

			/* Update joint probability distribution along internal->internal_node
			 * i represents parent of internal node
			 * j represents internal node
			 * k represents no of sites
			 */
			for(i=0; i<4; i++) {//parent
	
				for(j=0; j<4; j++) {//child			
					
					denominator = 0.0;
					sum = 0.0;
				
					for(k=0; k<noOfVariableSites; k++) {
					
						siteProb = jointProbDistribution[i][j]
									* siteSpecificParentChildNodeVal[k][i][j] ;
								
						/* Denominator is a double sum over {A,C,G,T}
						 * corresponding to internal node and its parent
						 */
						denominator = ((Double)probList.get(k)).doubleValue();
	
						sum += patternCountArray[k] * siteProb/denominator;					
						
					}	// end of FOR loop for no of sites
				
					//divide by N (= no_of_sites)
					//sum = sum/ noOfSites;
					
					//set new probability value for (i, j)
					updateMatrix[i][j] = sum;				
		
				} // end of FOR loop for internal node
										
			} // end of FOR loop for parent of internal node
			
			LagrangeMultipliers lagrangeEstimates = (LagrangeMultipliers) updateLagMultPerBranch.get(nodeName);

			double[] currentLambdaEst = lagrangeEstimates.getRowMultipliers();
			double[] currentDeltaEst = lagrangeEstimates.getColMultipliers();

			double[] updatedLambdaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 1);
			double[] updatedDeltaEst = getLambdaDeltaEst(currentLambdaEst, currentDeltaEst, updateMatrix, 2);
		
			lagrangeEstimates.setRowMultipliers(updatedLambdaEst);
			lagrangeEstimates.setColMultipliers(updatedDeltaEst);
			updateLagMultPerBranch.put(nodeName, lagrangeEstimates);

			//Get the revised estimates of Q-matrix
			for(int parent=0; parent<4; parent++) {
				for(int child=0; child<4; child++) {
					updateMatrix[parent][child] /= (updatedLambdaEst[parent] + updatedDeltaEst[child]);
				}
			}

			//Release memory			
			currentLambdaEst = null;
			currentDeltaEst = null;

			updatedLambdaEst = null;
			updatedDeltaEst = null;
			probList = null;
			
			bdUpdated = new BranchDetails();
			bdUpdated.setJointProbability(updateMatrix);
																			
			/* Updated Q-matrix values are store in a temporary TreeMap that changes
			 * with every iteration. Those branches where original and re-calculated
			 * Q-matrix values are different (as determined by the function 
			 * setUpdateOption), the branchList TreeMap is updated with the new
			 * Q-matrix values  
			 */
			updateBranchList.put(nodeName, bdUpdated);
			if(iterationOption != '2') setUpdateOption(nodeName, bd, bdUpdated);
            //setUpdateOption(nodeName, bd, bdUpdated);
	
			//updateMatrix = null;
			bd = null;
			bdUpdated = null;
					
			//Store the updated Q-values in the file 
			writeOutput(nodeName);
		
		} //end of OPTION
				
		return logLikelihood;	
	}
	
	/** This method computes the conditional probability P(x2|x1)
	 *
	 * @param parent	Nucleotide at parent node (wrt root node)
	 * @param child		Nucleotide at child node (wrt root node)
	 * @param direction	Direction of traversal
	 *					1 -> Forward direction from parent to child node
	 *					2 -> Reverse direction from child to parent node
	 * @param branch	Branch of phylogenetic tree
	 * @return			P(x1|x2)
	 */
	double pX1X2(int parent, int child, int direction, String branch) {
		BranchDetails bd;
		double marginalProb = 0.0; 
		double condProb = 0.0;
		
		//Obtain joint probability matrix for appropriate branch
		bd = (BranchDetails)branchList.get(branch); 				
		double[][] probDistribution = bd.getJointProbability();
		
		/* nucleotide = 0 => base "A"
		 * nucleotide = 1 => base "C"
		 * nucleotide = 2 => base "G"
		 * nucleotide = 3 => base "T"
		 */
		
		if(direction == 1) {
			/* Forward direction of traversal from parent node to child node
			 * So, sum over all columns i.e. child values are added
			 */
			marginalProb += probDistribution[parent][0];
			marginalProb += probDistribution[parent][1];
			marginalProb += probDistribution[parent][2];
			marginalProb += probDistribution[parent][3];		
		}	
		else {
			/* Reverse direction of traversal from child node to parent node
			 * So, sum over all rows i.e. parent values are added
			 */
			marginalProb += probDistribution[0][child];
			marginalProb += probDistribution[1][child];
			marginalProb += probDistribution[2][child];
			marginalProb += probDistribution[3][child];
		}			
			
		condProb = probDistribution[parent][child]/marginalProb;		
		
		return condProb;
	} 
	
	/**
	 * This method computes P(Ri|x1), the probability of reaching all the 
	 * leaf nodes connected to x1 i.e. x1 acts as the root node
	 *
	 * @param node			Node corresponding to x1
	 * @param base		 	Nucleotide character at node
	 * @param position		Nucleotide site
	 * @return				P(Ri|X1)
	 */
	double pRiX1(String node, int base, int position) {
		NodeStructure ns;
		ArrayList l1;	//ArrayList for sequence of childNode1
		ArrayList l2;	//ArrayList for sequence of childNode1
		int base1;		//base at specific position in l1
		int base2;		//base at specific position in l2
		String node1Type = null;	//Node type for child node1
		String node2Type = null;	//Node type for child node2
		double prob = 0.0;
		double probChildNode1 =0.0;
		double probChildNode2 =0.0;
		InternalNodeRiXi nRiXi;
		
		//Obtain the child nodes of node
		ns = (NodeStructure)phylogeneticTree.get(node);
		String childNode1 = ns.getChildNode1();
		String childNode2 = ns.getChildNode2();
		
		//Determine whether the child nodes are leaf nodes
		if(childNode1 != null) {
			ns = (NodeStructure)phylogeneticTree.get(childNode1);
			node1Type = ns.getNodeType();
		}
		else {
			probChildNode1 = 1;
		}
		
		if(childNode2 != null) {
			ns = (NodeStructure)phylogeneticTree.get(childNode2);
			node2Type = ns.getNodeType();
		}
		else {
			probChildNode2 = 1;	
		}
		
		if(probChildNode1 == 0) {
			if(node1Type.equals("LEAF") || node1Type.equals("ROOT")) {
				l1 = (ArrayList)treeSequences.get(childNode1);
				base1 = ((Integer)l1.get(position)).intValue();						
				l1 = null; //clear the memory occupied by l1

			   /* nucleotide at parent node = base
			 	* nucleotide at child node = base1
			 	* direction of traversal = fwd
			 	* branch is represented by childNode1
			 	*/							
				probChildNode1 = pX1X2(base, base1, 1, childNode1); 				
			}			
			else {
				
				if((probRiXi[position].size() > 0) 
						&& probRiXi[position].containsKey(childNode1)) {
					nRiXi = (InternalNodeRiXi)probRiXi[position].get(childNode1);
				}
				else {
					nRiXi = new InternalNodeRiXi();
				
					nRiXi.setNucleotideA(pRiX1(childNode1, 0, position)) ;
					nRiXi.setNucleotideC(pRiX1(childNode1, 1, position)) ;
					nRiXi.setNucleotideG(pRiX1(childNode1, 2, position)) ;
					nRiXi.setNucleotideT(pRiX1(childNode1, 3, position)) ;
				
					probRiXi[position].put(childNode1, nRiXi);
				}
			
				baseProb = nRiXi.getNucleotides();
				
				probChildNode1 = pX1X2(base, 0, 1, childNode1) * baseProb[0];							
				probChildNode1 += pX1X2(base, 1, 1, childNode1) * baseProb[1]; 
				probChildNode1 += pX1X2(base, 2, 1, childNode1) * baseProb[2]; 
				probChildNode1 += pX1X2(base, 3, 1, childNode1) * baseProb[3]; 
			}
		}
		
		if(probChildNode2 == 0) {
			if(node2Type.equals("LEAF") || node2Type.equals("ROOT")) {
				l2 = (ArrayList)treeSequences.get(childNode2);
				base2 = ((Integer)l2.get(position)).intValue();						
				l2 = null; //clear the memory occupied by l2

				/* nucleotide at parent node = base
				 * nucleotide at child node = {base1, base2}
				 * direction of traversal = fwd
				 * branches represented by {childNode1, childNode2}
			 	 */	
				probChildNode2 = pX1X2(base, base2, 1, childNode2);
			}
			else {
				
				if((probRiXi[position].size() > 0) 
						&& probRiXi[position].containsKey(childNode2)) {
					nRiXi = (InternalNodeRiXi)probRiXi[position].get(childNode2);
				}
				else {
					nRiXi = new InternalNodeRiXi();
				
					nRiXi.setNucleotideA(pRiX1(childNode2, 0, position)) ;
					nRiXi.setNucleotideC(pRiX1(childNode2, 1, position)) ;
					nRiXi.setNucleotideG(pRiX1(childNode2, 2, position)) ;
					nRiXi.setNucleotideT(pRiX1(childNode2, 3, position)) ;
				
					probRiXi[position].put(childNode2, nRiXi);
				}
				
				baseProb = nRiXi.getNucleotides();
				
				probChildNode2 = pX1X2(base, 0, 1, childNode2) * baseProb[0]; 
			 	probChildNode2 += pX1X2(base, 1, 1, childNode2) * baseProb[1];
			 	probChildNode2 += pX1X2(base, 2, 1, childNode2) * baseProb[2]; 
			 	probChildNode2 += pX1X2(base, 3, 1, childNode2) * baseProb[3]; 
			}	
		}
		
		prob = probChildNode1 * probChildNode2;
		
		return prob;		
	} 
	
	/**
	 * This method computes P(Si|x2) i.e. probability of reaching all the leaf
	 * nodes connected to T(j)
	 * @param node			Node corresponding to x2
	 * @param blockedNode	Node from which x2 was reached
	 * @param base		 	Nucleotide character at node x2
	 * @param position		Nucleotide site
	 * @return 				P(Si|X2)
	 */
	 double pSiX2(String node, String blockedNode, int base, int position) {
		NodeStructure ns;	 	
		String childNode = null;
		String parentNode;
		double pChildNode = 1.0;
		double pParentNode = 1.0;
		int base1;
		ArrayList l1;
		InternalNodeRiXi nRiXi;
		InternalNodeSiXi nSiXi;
		
	 	// Determine parent node and child nodes of "node"
	 	ns = (NodeStructure)phylogeneticTree.get(node);
	 	String t1 = ns.getChildNode1();
	 	String t2 = ns.getChildNode2();
	 	parentNode = ns.getParentNode();
	 	
	 	//consider only the parent node and non-blocked child node
	 	if((t1 != null) && (!t1.equals(blockedNode))) childNode = t1;	 	
	 	if((t2 != null) && (!t2.equals(blockedNode))) childNode = t2;	 		 	
	 	
	 	/* if child node is a leaf node, determine conditional probability
	 	 * else call function pRiX1 for {A,C,G,T} and sum over them
	 	 */
	 	if(childNode != null) {
	 		//determine if it is a leaf node
	 		ns = (NodeStructure)phylogeneticTree.get(childNode);
	 		if((ns.getNodeType()).equals("LEAF") || (ns.getNodeType()).equals("ROOT")) {
	 			
				l1 = (ArrayList)treeSequences.get(childNode);
				base1 = ((Integer)l1.get(position)).intValue();						
				l1 = null; //clear the memory occupied by l2

	 			/* parent = "base" at parent node
	 			 * child = base at child node i.e. "base1"
	 			 * direction of traversal = fwd
	 			 * branch = childNode
	 			 */
	 			pChildNode = pX1X2(base, base1, 1, childNode);	 			
	 		}	 				
	 		else {
	 			//consider all 4 values {A,C,G,T}
	 			
				if((probRiXi[position].size() > 0) 
					&& probRiXi[position].containsKey(childNode)) {
						
					nRiXi = (InternalNodeRiXi)probRiXi[position].get(childNode);
				}
				else {
					nRiXi = new InternalNodeRiXi();
				
					nRiXi.setNucleotideA(pRiX1(childNode, 0, position));
					nRiXi.setNucleotideC(pRiX1(childNode, 1, position));
					nRiXi.setNucleotideG(pRiX1(childNode, 2, position));
					nRiXi.setNucleotideT(pRiX1(childNode, 3, position));
					
					probRiXi[position].put(childNode, nRiXi);
				}
			
				baseProb = nRiXi.getNucleotides();
				double siteA = baseProb[0];
				double siteC = baseProb[1];
				double siteG = baseProb[2];
				double siteT = baseProb[3];

	 			pChildNode = pX1X2(base, 0, 1, childNode) * siteA;
	 			pChildNode += pX1X2(base, 1, 1, childNode) * siteC;
	 			pChildNode += pX1X2(base, 2, 1, childNode) * siteG;
	 			pChildNode += pX1X2(base, 3, 1, childNode) * siteT;
	 			
	 		}
	 	} 
	 	
	 	/* if parent node = leaf node, determine conditional probability
	 	 * in reverse direction. Else call the method pSiX2 recursively with 
	 	 * current "node" as blocked node 
	 	 */
	 	//System.out.println("Parent node = " + parentNode); 
	 	ns = (NodeStructure)phylogeneticTree.get(parentNode);
	 	
	 	if((ns.getNodeType()).equals("LEAF") || (ns.getNodeType()).equals("ROOT")) {

			l1 = (ArrayList)treeSequences.get(parentNode);
			base1 = ((Integer)l1.get(position)).intValue();						
			l1 = null; //clear the memory occupied by l2

	 		/* parent = "base1" 
	 		 * child = "base"
	 		 * direction of traversal = reverse
	 		 * branch = childNode
	 		 */

	 		//Obtain joint probability distribution for (base1, base)
	 		BranchDetails bd = (BranchDetails)branchList.get(node); 				
			double[][] probDistribution = bd.getJointProbability();
	 		
	 		pParentNode = probDistribution[base1][base];
	 			 		
	 	}
	 	else {

			if((probSiXi[position].size() > 0) 
				&& probSiXi[position].containsKey(parentNode + ":" + node)) {
					
				nSiXi = (InternalNodeSiXi)probSiXi[position].get(parentNode + ":" + node);
			}
			else {
				nSiXi = new InternalNodeSiXi();
			
				nSiXi.setNucleotideA(pSiX2(parentNode, node, 0, position));
				nSiXi.setNucleotideC(pSiX2(parentNode, node, 1, position));
				nSiXi.setNucleotideG(pSiX2(parentNode, node, 2, position));
				nSiXi.setNucleotideT(pSiX2(parentNode, node, 3, position));
				
				probSiXi[position].put(parentNode + ":" + node, nSiXi);
			}
		
			baseProb = nSiXi.getNucleotides();

	 		/* Parent node is an internal node so consider {A,C,G,T} values at
	 		 * node
	 		 */
	 		pParentNode = pX1X2(0, base,1,node) * baseProb[0];
	 		pParentNode += pX1X2(1, base,1,node) * baseProb[1]; 
	 		pParentNode += pX1X2(2, base,1,node) * baseProb[2];
	 		pParentNode += pX1X2(3, base,1,node) * baseProb[3];
	 						
	 	}	 	
	 	 
	 	 return (pParentNode * pChildNode);
	 }
	 
	/**
	 * This method computes P(G1j|x1)*P(G2j|x2) i.e. probability of reaching 
	 * the leaf nodes that are connected to T(j) and the probability of 
	 * reaching all the leaf nodes that are descendants of x2
	 *	
	 * @param node				Node 
	 * @param parentNode		Parent of node
	 * @param nodeBase			Nucleotide character at node
	 * @param parentNodeBase	Nucleotide character at parent node
	 * @param position			Nucleotide site
	 * @return					P(G1j)*P(G2j)
	 */
	 double pG1jG2j(String node, String parentNode, int nodeBase,
	 				int parentNodeBase, int position) {
	 					
	 	double pG2j = 1.0;
	 	double pG1j = 1.0;
	 
	 	/* Compute the probability of moving from "node" to leaf nodes that are 
	 	 * its descendants
	 	 */
	 	pG2j = pRiX1(node, nodeBase, position);
	 	 
	 	/* Compute the probability of moving from parent node i.e. T(j) to all
	 	 * the relevant leaf nodes
	 	 */						
		pG1j = pSiX2(parentNode, node, parentNodeBase, position);
		
		BranchDetails bd = (BranchDetails)branchList.get(parentNode);
		double[][] probDistribution = bd.getJointProbability();
		
		double sum = probDistribution[0][parentNodeBase]
					+ probDistribution[1][parentNodeBase]
					+ probDistribution[2][parentNodeBase]
					+ probDistribution[3][parentNodeBase];
		
		pG1j = pG1j/ sum;
		
		return (pG2j * pG1j);
	 	 
	 }
	 
	 /**
	  * This method computes the denominator for updation of probability
	  * involving two internal nodes
	  *
	  * @param nodeName		Internal node name
	  * @param parentNode	Parent of internal node
	  * @param position		Site position
	  * @return				Q(x1,x2) * P(G1j) * P(G2j)
	  */
	 double computeIntToIntDenominator(String nodeName, String parentNode,
	 									int position) {
	 	
	 	BranchDetails bd = (BranchDetails)branchList.get(nodeName);
	 	double jointProb[][] = bd.getJointProbability();
	 	double prob = 0.0;
	 	
		/* i represents parent of internal node
		 * j represents internal node
		 */
	 	for(int i=0; i<4; i++) {
	 		for(int j=0; j<4; j++) {
	 			prob += jointProb[i][j] 
	 						* pG1jG2j(nodeName, parentNode, j, i, position);
	 				
	 		} // end of FOR loop for internal node
	 	} // end of FOR loop for parent of internal node
	 	
	 	return prob;
	 }

	/**
	 * This method initializes the Q-matrix updation HashMap
	 *
	 * @param internalList		List of internal nodes
	 * @param leafList		    List of leaf nodes
	 */
	 void initializeUpdateOption(ArrayList internalList, ArrayList leafList, int noOfSites) {
	 	int count = internalList.size();
	 	
		for(int i=0; i<count; i++) {
			isUpdateReqd.put((String)internalList.get(i), new Boolean(true));			
			updateLagMultPerBranch.put((String)internalList.get(i), new LagrangeMultipliers(noOfSites));
		}
		
		count = leafList.size();
		
		for(int i=0; i<count; i++) {
			isUpdateReqd.put((String)leafList.get(i), new Boolean(true));	
			updateLagMultPerBranch.put((String)leafList.get(i), new LagrangeMultipliers(noOfSites));
		}

	 }

	/**
	 * This method determines whether in the next iteration Q-values should be
	 * updated for the branch determined by parent of "name" and "name"
	 *
	 * @param name			Node name
	 * @param bd			Previous Q-matrix
	 * @param bdUpdated		Current Q-matrix
	 */
	 void setUpdateOption(String name, BranchDetails bd
	 						, BranchDetails bdUpdated) {
	 	double result = 0.0;
	 	double qCurrArray[][] = bd.getJointProbability(); 
	 	double qUpdateArray[][] = bdUpdated.getJointProbability();
	 	
	 	for(int i=0; i<4; i++) {
	 		for(int j=0; j<4; j++) {
	 			result += Math.pow((qCurrArray[i][j] - qUpdateArray[i][j]), 2.0);
	 		}
	 	}
	 	
	 	// Do not update if sum of squares is less than user-specified value
	 	if(result<minValue) {
			isUpdateReqd.put(name, new Boolean(false));	 		
	 	}
	 }
	 
	 /** This method is used to generate random Q-matrices along the edge
	  *  (root_node, child_node) for the phylogenetic tree
	  * 
	  * @param root		Root node name
	  */
	  void generateRootQMatrix(String root) {
	  		BranchDetails bdGen = new BranchDetails();
	  		nsGen = (NodeStructure)phylogeneticTree.get(root);
	  		
	  		String childNode1 = nsGen.getChildNode1();
	  		String childNode2 = nsGen.getChildNode2();
	  		
	  		//Generate probabilities for A, C, G, and T
	  		Random r = new Random();
	  		
	  		double pA = r.nextDouble();
	  		double pC = r.nextDouble();
	  		double pG = r.nextDouble();
	  		double pT = r.nextDouble();
	  		double sum = pA + pC + pG + pT;
	  		
	  		pA /= sum;
	  		pC /= sum;
	  		pG /= sum;
	  		pT /= sum;
	  		
	  		//Generate matrix with known values of p(A), p(C), p(G), p(T)
	  		//qGenArray = qMatGen.generateMatrix(pA, pC, pG, pT, 1); //Commented 18-Aug-06	  		
			qGenArray = qMatGen.generateMatrix(marginalProbVector[0], marginalProbVector[1], marginalProbVector[2], marginalProbVector[3], 1); //Added 18-Aug-06 	  		
	  		
	  		//Generate matrix with random values of p(A), p(C), p(G), p(T)
	  		//qGenArray = qMatGen.generateMatrix();
	  		
	  		bdGen.setJointProbability(qGenArray);
	  		
	  		pA = qGenArray[0][0] + qGenArray[1][0] + qGenArray[2][0] + qGenArray[3][0];
	  		pC = qGenArray[0][1] + qGenArray[1][1] + qGenArray[2][1] + qGenArray[3][1];
	  		pG = qGenArray[0][2] + qGenArray[1][2] + qGenArray[2][2] + qGenArray[3][2];
	  		pT = qGenArray[0][3] + qGenArray[1][3] + qGenArray[2][3] + qGenArray[3][3];
	  		
	  		if(childNode1 != null) {		
				branchList.put(childNode1, bdGen);

		  		//Compute Q matrices for the remaining nodes
		  		generateRandomQMatrices(childNode1, pA, pC, pG, pT);
	  		}
	  		else {
	  			branchList.put(childNode2, bdGen);
	  			generateRandomQMatrices(childNode2, pA, pC, pG, pT);
	  		}
	  		
	  }

	 /** This method is used to generate random Q-matrices along all the 
	  *  remaining edges for the phylogenetic tree
	  * 
	  * @param childNode		Branch (parent_node, child_node)
	  * @param pA				Probability of nucleotide A
	  * @param pC				Probability of nucleotide C
	  * @param pG				Probability of nucleotide G
	  * @param pT				Probability of nucleotide T
	  */
	  
	  void generateRandomQMatrices(String childNode, double  pA, double pC, 
	  								double pG, double pT) {
	  		/* Cannot be taken out of this method else all branchDetails will be
	  		 * initialized to the last Q-matrix that is generated
	  		 */
	  		BranchDetails bdGen = new BranchDetails();
	  		double dA, dC, dG, dT;
	  									
	  		nsGen = (NodeStructure)phylogeneticTree.get(childNode);
	  		
	  		String childNode1 = nsGen.getChildNode1();
	  		String childNode2 = nsGen.getChildNode2();
	  		
	  		if(childNode1 != null) {
	  			
				//Generate matrix with known values of p(A), p(C), p(G), p(T)
		  		qGenArray = qMatGen.generateMatrix(pA, pC, pG, pT, 1); 	  		
		  		
		  		//Generate matrix with random values of p(A), p(C), p(G), p(T)
		  		//qGenArray = qMatGen.generateMatrix();
		  		
		  		bdGen.setJointProbability(qGenArray);
		  		branchList.put(childNode1, bdGen);
		  		
		  		dA = qGenArray[0][0] + qGenArray[1][0] + qGenArray[2][0] 
		  				+ qGenArray[3][0];
	  			dC = qGenArray[0][1] + qGenArray[1][1] + qGenArray[2][1] 
	  					+ qGenArray[3][1];
	  			dG = qGenArray[0][2] + qGenArray[1][2] + qGenArray[2][2] 
	  					+ qGenArray[3][2];
	  			dT = qGenArray[0][3] + qGenArray[1][3] + qGenArray[2][3] 
	  					+ qGenArray[3][3];		  		
	  			
	  			generateRandomQMatrices(childNode1, dA, dC,dG, dT);
	  		}
	  		
	  		if(childNode2 != null) {
	  			
	  			/* Needed to ensure that same branchDetails are not copied to both 
	  			 * the child nodes
	  			 */
	  			bdGen = new BranchDetails();
	  			
		  		//Generate matrix with known values of p(A), p(C), p(G), p(T)
		  		qGenArray = qMatGen.generateMatrix(pA, pC, pG, pT, 1); 	  		
		  		
		  		//Generate matrix with random values of p(A), p(C), p(G), p(T)
		  		//qGenArray = qMatGen.generateMatrix();
		  		
		  		bdGen.setJointProbability(qGenArray);
		  		branchList.put(childNode2, bdGen);	
		  			  			
		  		dA = qGenArray[0][0] + qGenArray[1][0] + qGenArray[2][0] 
		  				+ qGenArray[3][0];
	  			dC = qGenArray[0][1] + qGenArray[1][1] + qGenArray[2][1] 
	  					+ qGenArray[3][1];
	  			dG = qGenArray[0][2] + qGenArray[1][2] + qGenArray[2][2] 
	  					+ qGenArray[3][2];
	  			dT = qGenArray[0][3] + qGenArray[1][3] + qGenArray[2][3] 
	  					+ qGenArray[3][3];		  		
	  			
	  			generateRandomQMatrices(childNode2, dA, dC,dG, dT);
	  		}	  	
	  }
	  
	 /** This method is used to write the initial Q-matrices value to a file
	  * 
	  * @param intNodes			List of internal nodes
	  * @param leafNodes		List of leaf nodes
	  */
	  void generateInitQMatricesFile(ArrayList intNodes, ArrayList leafNodes
	  								, String initQMatrixFile) {
	  	
	  	BranchDetails bdGen = new BranchDetails();
		               
        //Determine the last occurence of the character '/' e.g C:/My_documents/Input/
        int slashIndex = destFilePath.lastIndexOf('/');
        String filePath = destFilePath.substring(0, slashIndex+1);
                
		try{

	  		FileWriter fwq = new FileWriter(filePath + initQMatrixFile);
	  		String nodeName;

	  		for(int i=0; i<intNodes.size(); i++) {
	  			nodeName = (String)intNodes.get(i);
	  			
	  			bdGen = (BranchDetails)branchList.get(nodeName);
	  			qGenArray = bdGen.getJointProbability();
	  			
	  			fwq.write("Internal Node = " + nodeName + "\n");
	  			
	  			for(int j=0; j<4; j++) {
	  				for(int k=0; k<4; k++) {
	  					fwq.write(qGenArray[j][k] + "\t");
	  				}
	  				fwq.write("\n");
	  			}
	  			
	  		} //end of internal nodes list
	  		
	  		for(int i=0; i<leafNodes.size(); i++) {
	  			nodeName = (String)leafNodes.get(i);
	  			bdGen = (BranchDetails)branchList.get(nodeName);
	  			qGenArray = bdGen.getJointProbability();
	  			
	  			fwq.write("Leaf Node = " + nodeName + "\n");
	  			
	  			for(int j=0; j<4; j++) {
	  				for(int k=0; k<4; k++) {
	  					fwq.write(qGenArray[j][k] + "\t");
	  				}
	  				fwq.write("\n");
	  			}
	  			
	  		} //end of leaf nodes list
	  		
	  		fwq.close();
	  		
	  	} catch(Exception e) {
	  		System.out.println("Error occured while writing to Q-Matrix File "
	  							+ e);
	  	}	
	  }
	  
	/**	This method writes the converged Q-values to the output file
	 *
	 *	@param	updateList			The branchList TreeMap structure that will
	 *								be copied to the updateBranchList data 
	 *								structure 		
	 */	
	void writeFinalQMatrices(TreeMap updateList) {	
		String nodeName;
		
		updateBranchList = updateList;
		
		for(int internalNodeCount=0; internalNodeCount < intNodesList.size();
		 	internalNodeCount++) {						
				nodeName = (String)intNodesList.get(internalNodeCount);	
				writeOutput(nodeName);
		}	

		for(int leafNodeCount=0; leafNodeCount < leafNodesList.size();
		 	leafNodeCount++) {
				nodeName = (String)leafNodesList.get(leafNodeCount);	
				writeOutput(nodeName);
		}	
		
	}  

	void writeQMatFile(TreeMap updateList, int dataSetCount) {	

		int index = srcFilePath.lastIndexOf('/');

		String nodeName;
		String data;
		String qMatFileName = srcFilePath.substring(0, index) + "/Q_Set_" + dataSetCount + ".txt";

		BranchDetails bd; 
		double[][] probDistribution;		

		try{
	
			FileWriter qMatWriter = new FileWriter(qMatFileName);

			for(int internalNodeCount=0; internalNodeCount < intNodesList.size();
			 	internalNodeCount++) {						
					nodeName = (String)intNodesList.get(internalNodeCount);	
			
					//writeOutput(nodeName);
					bd = (BranchDetails)updateBranchList.get(nodeName);
					probDistribution = bd.getJointProbability();

					for(int i=0; i<4; i++) {
						for(int j=0; j<4; j++) {
							data = probDistribution[i][j] + "\t";
							qMatWriter.write(data);
						}  // end of inner FOR loop

						qMatWriter.write("\n");	
					}// end of outer FOR loop

					qMatWriter.write("\n");	//A blank line separates successive q-matrices
			}	

			for(int leafNodeCount=0; leafNodeCount < leafNodesList.size();
			 	leafNodeCount++) {
					nodeName = (String)leafNodesList.get(leafNodeCount);	

					//writeOutput(nodeName);
					bd = (BranchDetails)updateBranchList.get(nodeName);
					probDistribution = bd.getJointProbability();

					for(int i=0; i<4; i++) {
						for(int j=0; j<4; j++) {
							data = probDistribution[i][j] + "\t";
							qMatWriter.write(data);
						}  // end of inner FOR loop

						qMatWriter.write("\n");	
					}// end of outer FOR loop

					qMatWriter.write("\n");	//A blank line separates successive q-matrices			}	
			}
	
			qMatWriter.close();

		}catch(Exception e) {
			e.printStackTrace();
		}
		
	}  	

	/**	This method is used to compute the log likelihood values and the Q-matrices
	 *	for each iteration
	 *
	 *	@param	rootNode			Root node of the phylogenetic tree
	 *	@param 	option				1 -> Compute both log likelihood and Q-values
	 *								2 -> Compute only log likelihood values
	 *	@param	iterationOption		Determine whether to compute the square of 
	 *								the difference between original and new Q-matrix
	 *								1 -> Compute the difference
	 *								2 -> Do not compute
	 *
	 */
	double iterateLogLikelihoodAndQValues(String rootNode, int option,
											char iterationOption) {
		
		boolean foundTopIntNode = false;
		double logLikelihoodValue = 0.0;
		String nodeName;
		String parentNode;
		String parentOfLeafNode;
		NodeStructure ns;

		//Initialize RiXi and SiXi hashmaps
		for(int i=0; i<noOfVariableSites; i++) {
			probRiXi[i] = new TreeMap();
			probSiXi[i] = new TreeMap();
		}
		
		foundTopIntNode = false;
		
		if(option ==1) {
			if(showOutput) {
				//System.out.println("Iteration Number = " + noOfIterations);	
			}
			
			writeOutput(noOfIterations++);
		}	

		/* Compute log likelihood and Q-values for all internal nodes
		 * For T(j) = 0, use method computeValuesIntToLeaf 
		 * else use computeValuesIntToInt 
		 */
		for(int internalNodeCount=0; internalNodeCount < intNodesList.size();
		 	internalNodeCount++) {				
		
			nodeName = (String)intNodesList.get(internalNodeCount);			
			
			ns = (NodeStructure)phylogeneticTree.get(nodeName);
			parentNode = ns.getParentNode();
			
			//if T(j)=0, pass root node and child node name to method
			if(!foundTopIntNode) {					
				if(parentNode.equals(rootNode)) {
					foundTopIntNode = true;
					
					/* Determine whether to compute log likelihood
					 * for a given node. If the updated Q-value is less than 
					 * minValue, then do not compute for that branch
					 */
					if((((Boolean)isUpdateReqd.get(nodeName)).booleanValue()) 
						|| (option == 2)){

						logLikelihoodValue = computeValuesRootToInt(nodeName,
											rootNode, option, iterationOption);
					} 						
				}
				else {
					if((((Boolean)isUpdateReqd.get(nodeName)).booleanValue()) 
						|| (option == 2)){

						logLikelihoodValue = computeValuesIntToInt(nodeName,
										 parentNode, option, iterationOption, false);
					} 						
				}
			}						
			else {
				/* if the internal node does not satisfy T(j) = 0, pass 
				 * node name and the parent node name
				 */						 
					if((((Boolean)isUpdateReqd.get(nodeName)).booleanValue()) 
						|| (option == 2)){

						logLikelihoodValue = computeValuesIntToInt(nodeName,
											 parentNode, option, iterationOption, false);
				 }						
			}
			
		 } // end of FOR loop for internal nodes

		/* Compute log likelihood and Q-values for all leaf nodes excluding
		 * the root node. Use formula 3	
		 */			 
		for(int leafNodeCount=0; leafNodeCount < leafNodesList.size();
		 	leafNodeCount++) {
		
			nodeName = (String)leafNodesList.get(leafNodeCount);

			//Obtain parent of leaf node
			ns = (NodeStructure)phylogeneticTree.get(nodeName);				
			parentOfLeafNode = ns.getParentNode();

			if((((Boolean)isUpdateReqd.get(nodeName)).booleanValue()) 
				|| (option == 2)){

				logLikelihoodValue = computeValuesIntToLeaf(parentOfLeafNode
									, nodeName, option, iterationOption);
			}
		 
		} // end of FOR loop for leaf nodes 

       return logLikelihoodValue;
	}	  
	
	
	void openSiteLogLikelihoodValueFile() {
		int lastIndex = destFilePath.lastIndexOf("/");
		String tPath = destFilePath.substring(0, lastIndex) + "/site.out";
		
		try{
			fSiteW = new FileWriter(tPath);
		} catch(Exception e) {
			System.out.println("Error opening site output file " + e);
		}	
	}
	
	void writeSiteLogLikelihoodValue(int siteNumber, double siteProb) {
		String data = "";
		
		if(siteNumber == 1) {
			data = "Log Likelihood value at site number  " + siteNumber
						+ " = " + Math.log(siteProb) + "\n"; 
		}
		else {
			data = Math.log(siteProb) + "\n"; 
		}
		
		try{
			fSiteW.write("\t\t" + siteNumber + "\t" + data);		
		} catch(Exception e) {
			System.out.println("Error occured while entering site specific log " 
								+ " likelihood value ");
		}
		
	}
	
	void closeSiteLogLikelihoodValueFile() {
		try{
			fSiteW.close();
		} catch(Exception e) {
			System.out.println("Error closing site output file " + e);
		}	
	}
	
	double[] getMarginalProb(String speciesSeq, double[] avg, int sitesCount, int speciesCount) {
		
		int[] baseCount = new int[4];
		double[] prob = new double[4];
		
		for(int i=0; i<sitesCount; i++) {
			switch(speciesSeq.charAt(i)) {
				case 'A': baseCount[0]++; break;
				case 'C': baseCount[1]++; break;
				case 'G': baseCount[2]++; break;
				case 'T': baseCount[3]++;
			}			
			
		} // all sites considered	
			
		for(int i=0; i<4; i++) prob[i] = avg[i] + baseCount[i]*1.0/sitesCount;
		
		return prob;
		
	}
	
	double estimateQMatrices(String rootNode) {

		boolean areQMatsOptimized = false;	

		while(!areQMatsOptimized) {
			
			iterateLogLikelihoodAndQValues(rootNode, 1, '1');
						
			/* Update Joint Probability Distribution values for edges that have 
			 * been modified 
			 */
			Set set = updateBranchList.entrySet();
			Iterator itr = set.iterator();
			Map.Entry me;
			String listNodeName;
			BranchDetails listBD;
			double listdArray[][] = new double[4][4];
				
			while(itr.hasNext()) {
				me = (Map.Entry) itr.next();
				listNodeName = (String)me.getKey();
				listBD = (BranchDetails)me.getValue();
				listdArray = listBD.getJointProbability();
					
				branchList.put(listNodeName, new BranchDetails(listdArray));
			}
	
			if(updateBranchList.size() == 0) areQMatsOptimized = true;
			updateBranchList = new TreeMap();

		} //Q-values optimized for variable sites

		double logL = iterateLogLikelihoodAndQValues(rootNode, 2, '1');

		return logL;
	}

	double[] getLambdaDeltaEst(double[] cLambdaEst, double[] cDeltaEst, double[][] uMatrix, int option) {

		double[] lagrangeEst = new double[4];

		if(option == 1) {

			for(int i=0; i<4; i++) {
				lagrangeEst[i] = 0;

				for(int j=0; j<4; j++) 
					lagrangeEst[i] += uMatrix[i][j]/(cLambdaEst[i] + cDeltaEst[j]);
			
				lagrangeEst[i] = lagrangeEst[i] * cLambdaEst[i] / marginalProbVector[i];

			}//Outer loop ends

		} //Lambda estimates obtained
		
		//Option == 2, Delta estimate
		if(option == 2) {

			for(int j=0; j<4; j++) {
				lagrangeEst[j] = 0;

				for(int i=0; i<4; i++) 
					lagrangeEst[j] += uMatrix[i][j]/(cLambdaEst[i] + cDeltaEst[j]);
		
				lagrangeEst[j] = lagrangeEst[j] * cDeltaEst[j] / marginalProbVector[j];

			}//Outer loop ends

		} //Delta estimates obtained		

		return lagrangeEst;		
	}

	void updateMargProbVector() {

		double lagMult = 0;

		for(int i=0; i<4; i++) lagMult += modifiedMargProb[i];

		for(int i=0; i<4; i++) {
			marginalProbVector[i] = modifiedMargProb[i]/ lagMult;
			//System.out.print(marginalProbVector[i] + ", ");
		}

		//System.out.println("");
		//for(int i=0; i<4; i++) modifiedMargProb[i] = 0;
	}

}	
