#! /usr/bin/env python3

"""Extract the Magma code from a literate Magma script

        MagmaCode.py
        D E Taylor
        21 July 2002
        Revised for python 3 3 October 2023

 A literate Magma script consists of narrative interspersed with
 executable Magma code.  Each code line begins with > 
 This program extracts the lines beginning with > and then removes the
 prompt (>) and one following blank (if it exists).

 Lines beginning with %% at the beginning of the file are turned into
 Magma comments by replacing the %% with // and then copied to the output
 Comment processing stops when a line not beginning with %% is encountered.

 A line beginning with %%stop halts the output of Magma code. It is resumed
 by a line beginning with %%start
 
 The first version of this program assumed that every initial > was
 followed by a space and therefore simply stripped the first two characters
 from each code line.

 2014-10-14: Allow directives within the TeX file which redirect the
 the code to a specified file.  The file name must be supplied on a line
 beginning with %%file: <filename> after comment processing is complete.

 2015-12-03: Allow multiple auxilliary files. The syntax is
 %%file: <filename> | tag
 In order to remain compatible with old code, the default tag is 'alt'

 A line beginning with %%main switches to the primary file.
 A line beginning with %%alt switches to the alternate file, generally
   used for test code.
 A line beginning with %%aux: tag switches to the auxilliary file 'tag'.

 
       Command line options:

          -d  debug

        Requires:
          Python 3
"""
import sys, getopt, time

VERSION = "MagmaCode 3.0"
DEBUG = 0
SEPHEAD = '\n//' + '='*78 + '\n'
SEPTAIL = '//' + '-'*78 + '\n'

def main():
  global DEBUG
  def fail(msg = None):
    if msg:
      print(msg)
    print('Usage: %s [-d] <script>' % sys.argv[0])
    sys.exit(1)

  if len(sys.argv) < 2:
    fail()

  try:
    opts, files = getopt.getopt(sys.argv[1:],'d')
  except getopt.GetoptError as e:
    print(e, file=sys.stderr)
    sys.exit(1)
    
  if not files:
    fail()
  for opt in opts:
    if opt[0] == '-d':
      DEBUG = 1

  infile = files[0]
  path = infile.split('.')
  if len(path) > 1:
    ext = path.pop()
  else:
    ext = ''

  if ext == "m" or ext == "M":
    fail("A literate script should not have a .m extension")

  path.append('m')
  mainfile = '.'.join(path)

  if DEBUG:
    print("[In]  = "+infile, file=sys.stderr)
    print("[Out] = "+mainfile, file=sys.stderr)

  try:
    f = open(infile,'r')
    primary = open(mainfile,'w')
  except IOError as e:
    print(e, file=sys.stderr)
    sys.exit(1)

  print("Primary:   "+mainfile, file=sys.stderr)

  # Start out writing to the main file
  out = primary
  # At this point there is no alternate file
  alt = None

  out.write('// Extracted from '+
    infile+' by '+VERSION+' on '+
    time.asctime(time.localtime(time.time()))+'\n//\n')

  for line in f:
    line = line.lstrip()
    if line.startswith('%%'):
      out.write('//'+line[2:])
    else:
      break

  out.write('\n')

  auxfiles = {}
  incode = 0
  active = True
  for line in f:
    if line.lstrip().startswith('\\end{document}'):
      break
    elif line.startswith('>'):
      incode = 1
      line = line[1:]
      if line.startswith(' '):
        line = line[1:]
      if active:
        if line.lstrip().startswith('intrinsic'):
          out.write(SEPHEAD)
        out.write(line)
        if line.lstrip().startswith('end intrinsic'):
          out.write(SEPTAIL)
    elif line.startswith('%%stop'):
      active = False
    elif line.startswith('%%start'):
      active = True
    elif line.startswith('%%main'):
      out = primary
    elif line.startswith('%%alt'):
      out = auxfiles["alt"]
    elif line.startswith('%%aux:'):
      fn = line[6:].strip()
      if fn in auxfiles:
        out = auxfiles[fn]
      else:
        print('Error: %s has not been declared' % fn, file=sys.stderr)
    elif line.startswith('%%file:'): 
      names = line[7:].split('|')
      fn = names[0].strip()
      tag = "alt" if len(names) == 1 else names[1].strip()
      if tag in auxfiles:
        print('Error: duplicate declaration of '+tag, file=sys.stderr)
      else:
        try:
          auxfiles[tag] = open(fn,'w')
        except IOError as e:
          print(e, file=sys.stderr)
          sys.exit(1)
        print('Secondary: '+fn, file=sys.stderr)
    elif line.startswith('%//'):
      incode = 1
      line = line[1:]
      if active:
        out.write(line)
    elif incode:
      if active:
        out.write('\n')
      incode = 0

  f.close()
  primary.close()
  for fh in list(auxfiles.values()):
    fh.close()

if __name__ == '__main__':
  main()

